package cz.gopay.sdk.ui

import android.app.Activity
import android.view.WindowManager
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.platform.LocalView
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import cz.gopay.sdk.GopaySDK
import cz.gopay.sdk.model.CardData
import cz.gopay.sdk.model.CardTokenResponse
import cz.gopay.sdk.ui.utils.CardNumberInputValidator
import cz.gopay.sdk.ui.utils.CardNumberMaskedVisualTransformation
import cz.gopay.sdk.ui.utils.CardNumberVisualTransformation
import cz.gopay.sdk.ui.utils.CardValidator
import cz.gopay.sdk.ui.utils.CvvMaskedVisualTransformation
import cz.gopay.sdk.ui.utils.CvvValidator
import cz.gopay.sdk.ui.utils.ExpirationDateInputValidator
import cz.gopay.sdk.ui.utils.ExpirationDateVisualTransformation
import kotlinx.coroutines.launch

/**
 * Result of card tokenization operation
 */
sealed class TokenizationResult {
    data class Success(val tokenResponse: CardTokenResponse) : TokenizationResult()
    data class Error(val message: String, val exception: Throwable? = null) : TokenizationResult()
}

/**
 * Input field configuration for custom labels and helper texts
 */
data class InputFieldConfig(
    val label: String,
    val helperText: String? = null,
    val errorText: String? = null,
    val hasError: Boolean = false,
    val placeholder: String? = null
)

/**
 * Configuration for all input fields in the payment form
 */
data class PaymentFormInputs(
    val cardNumber: InputFieldConfig = InputFieldConfig(label = "Card Number", placeholder = "1234 1234 1234 1234"),
    val expirationDate: InputFieldConfig = InputFieldConfig(label = "MM/YY", placeholder = "MM/YY"),
    val cvv: InputFieldConfig = InputFieldConfig(label = "CVV", placeholder = "123")
)

/**
 * Helper function to format expiration date digits for validation
 */
private fun formatExpirationForValidation(digits: String): String {
    return when {
        digits.length <= 2 -> digits
        else -> "${digits.substring(0, 2)}/${digits.substring(2)}"
    }
}

/**
 * Helper function to parse expiration date
 */
private fun parseExpirationDate(expirationDate: String): Pair<Int, Int>? {
    val parsed = CardValidator.parseExpirationDate(expirationDate) ?: return null
    val month = parsed.first.toIntOrNull() ?: return null
    val year = parsed.second.toIntOrNull() ?: return null
    if (month < 1 || month > 12) return null
    return Pair(month, 2000 + year)
}

/**
 * Theme interface for customizing the appearance of the PaymentCardForm
 */
data class PaymentCardFormTheme(
    // Text styles
    val labelTextStyle: TextStyle = TextStyle(color = Color.Gray, fontSize = 14.sp),
    val inputTextStyle: TextStyle = TextStyle(fontSize = 16.sp),
    val helperTextStyle: TextStyle = TextStyle(color = Color.Gray, fontSize = 12.sp),
    val errorTextStyle: TextStyle = TextStyle(color = Color.Red, fontSize = 12.sp),
    val loadingTextStyle: TextStyle = TextStyle(color = Color.Gray, fontSize = 14.sp),
    
    // Colors
    val inputBorderColor: Color = Color.Gray,
    val inputErrorBorderColor: Color = Color.Red,
    val inputBackgroundColor: Color = Color.White,
    
    // Sizes and shapes
    val inputBorderWidth: Dp = 1.dp,
    val inputShape: Shape = RoundedCornerShape(4.dp),
    val inputPadding: PaddingValues = PaddingValues(12.dp),
    val fieldSpacing: Dp = 2.dp,
    val groupSpacing: Dp = 16.dp
)

/**
 * A secure payment card form that handles card data input and tokenization.
 * This component never exposes raw card data to the parent - it handles tokenization internally
 * and only returns the secure token to the caller.
 * 
 * Security Features:
 * - When debug mode is enabled in the SDK configuration, this form automatically sets FLAG_SECURE
 *   on the activity window to prevent screen capture and screenshots
 * - All card data is validated and tokenized securely within the SDK
 * - Raw card data is never exposed to the parent component
 *
 * @param onTokenizationComplete Callback called when tokenization completes (success or error)
 * @param modifier Modifier for the form layout
 * @param onFormReady Callback that provides a submit function for external triggering
 * @param onValidationError Callback called when validation errors occur
 * @param inputFields Configuration for input field labels, helper texts, and error states
 * @param theme Theme configuration for customizing the form appearance
 * @param permanent Whether to save the card for permanent usage (default: false)
 */
@Composable
fun PaymentCardForm(
    onTokenizationComplete: (TokenizationResult) -> Unit,
    modifier: Modifier = Modifier,
    onFormReady: ((suspend () -> TokenizationResult) -> Unit)? = null,
    onValidationError: ((CardValidator.CardValidationResult) -> Unit)? = null,
    inputFields: PaymentFormInputs = PaymentFormInputs(),
    theme: PaymentCardFormTheme = PaymentCardFormTheme(),
    permanent: Boolean = false
) {
    // Store clean input values (digits only)
    var cardNumberDigits by remember { mutableStateOf("") }
    var expirationDateDigits by remember { mutableStateOf("") }
    var cvv by remember { mutableStateOf("") }
    var isCardNumberFocused by remember { mutableStateOf(false) }
    var isCvvFocused by remember { mutableStateOf(false) }
    val isCardNumberValid = CardValidator.validateCardNumber(cardNumberDigits).isValid

    // FocusRequesters for each field
    val cardNumberFocusRequester = remember { FocusRequester() }
    val expirationFocusRequester = remember { FocusRequester() }
    val cvvFocusRequester = remember { FocusRequester() }
    val coroutineScope = rememberCoroutineScope()

    // Get the view to access the window for FLAG_SECURE
    val view = LocalView.current

    // Create the submit function
    val submitCardData: suspend () -> TokenizationResult = {
        submitCardDataImpl(
            cardNumberDigits = cardNumberDigits,
            expirationDateDigits = expirationDateDigits,
            cvv = cvv,
            permanent = permanent,
            onTokenizationComplete = onTokenizationComplete,
            onValidationError = onValidationError
        )
    }

    // Provide the submit function to parent if callback is provided
    LaunchedEffect(Unit) {
        onFormReady?.invoke(submitCardData)
    }

    // Set FLAG_SECURE for production builds to prevent screen capture. Disable for debug builds.
    LaunchedEffect(Unit) {
        try {
            val sdk = GopaySDK.getInstance()
            if (!sdk.isDebugEnabled()) {
                val activity = view.context as? Activity
                activity?.window?.addFlags(WindowManager.LayoutParams.FLAG_SECURE)
            }
        } catch (e: Exception) {
            // Silently handle any errors accessing the window
            // This prevents crashes if the view context is not an Activity
        }
    }

    Column(
        modifier = modifier
            .fillMaxWidth(),
        verticalArrangement = Arrangement.spacedBy(theme.fieldSpacing)
    ) {
        // Card Number Input
        LabeledInputField(
            value = cardNumberDigits,
            onValueChange = { newValue ->
                val validated = CardNumberInputValidator.validateInput(newValue)
                cardNumberDigits = validated
                if (CardNumberInputValidator.isValidLength(validated) && CardNumberInputValidator.isValidLuhn(validated)) {
                    coroutineScope.launch {
                        expirationFocusRequester.requestFocus()
                    }
                }
            },
            config = LabeledInputFieldConfig(
                label = inputFields.cardNumber.label,
                error = inputFields.cardNumber.errorText,
                helperText = inputFields.cardNumber.helperText,
                placeholder = inputFields.cardNumber.placeholder,
                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                visualTransformation = if (!isCardNumberFocused && isCardNumberValid) CardNumberMaskedVisualTransformation() else CardNumberVisualTransformation(),
                textFieldModifier = Modifier
                    .focusRequester(cardNumberFocusRequester)
                    .onFocusChanged { isCardNumberFocused = it.isFocused }
            ),
            theme = theme
        )
        // Expiration Date and CVV Row
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.spacedBy(theme.groupSpacing)
        ) {
            // Expiration Date
            LabeledInputField(
                value = expirationDateDigits,
                onValueChange = { newValue ->
                    val validated = ExpirationDateInputValidator.validateInput(newValue)
                    expirationDateDigits = validated
                    if (validated.length == 4) {
                        coroutineScope.launch {
                            cvvFocusRequester.requestFocus()
                        }
                    }
                },
                config = LabeledInputFieldConfig(
                    label = inputFields.expirationDate.label,
                    error = inputFields.expirationDate.errorText,
                    helperText = inputFields.expirationDate.helperText,
                    placeholder = inputFields.expirationDate.placeholder,
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    visualTransformation = ExpirationDateVisualTransformation(),
                    textFieldModifier = Modifier.focusRequester(expirationFocusRequester)
                ),
                modifier = Modifier.weight(1f),
                theme = theme
            )
            // CVV Input
            LabeledInputField(
                value = cvv,
                onValueChange = { newValue ->
                    cvv = CvvValidator.validateInput(newValue, cvv)
                },
                config = LabeledInputFieldConfig(
                    label = inputFields.cvv.label,
                    error = inputFields.cvv.errorText,
                    helperText = inputFields.cvv.helperText,
                    placeholder = inputFields.cvv.placeholder,
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    visualTransformation = if (isCvvFocused) VisualTransformation.None else CvvMaskedVisualTransformation(),
                    textFieldModifier = Modifier
                        .focusRequester(cvvFocusRequester)
                        .onFocusChanged { isCvvFocused = it.isFocused }
                ),
                modifier = Modifier.weight(1f),
                theme = theme
            )
        }
    }
}

private suspend fun submitCardDataImpl(
    cardNumberDigits: String,
    expirationDateDigits: String,
    cvv: String,
    permanent: Boolean,
    onTokenizationComplete: (TokenizationResult) -> Unit,
    onValidationError: ((CardValidator.CardValidationResult) -> Unit)?
): TokenizationResult {
    return try {
        // Validate all fields first
        val validation = CardValidator.validateCard(
            cardNumber = cardNumberDigits,
            expirationDate = formatExpirationForValidation(expirationDateDigits),
            cvv = cvv
        )

        if (!validation.isAllValid) {
            // Call validation error callback if provided
            onValidationError?.invoke(validation)
            throw IllegalArgumentException("Please fix the validation errors")
        }

        // Parse expiration date using utility
        val formattedExpDate = formatExpirationForValidation(expirationDateDigits)
        val expParts = parseExpirationDate(formattedExpDate)
            ?: throw IllegalArgumentException("Invalid expiration date format. Use MM/YY")

        val (expMonth, expYear) = expParts

        // Create card data - this stays within the SDK
        val cardData = CardData(
            cardPan = cardNumberDigits,
            expMonth = expMonth.toString().padStart(2, '0'),
            expYear = expYear.toString(),
            cvv = cvv
        )

        // Use SDK to tokenize card (using existing method)
        val tokenResponse = GopaySDK.getInstance().tokenizeCard(cardData, permanent)

        val result = TokenizationResult.Success(tokenResponse)
        onTokenizationComplete(result)
        result
    } catch (e: Exception) {
        val result = TokenizationResult.Error(
            message = e.message ?: "Card tokenization failed",
            exception = e
        )
        onTokenizationComplete(result)
        result
    }
} 