/**
 * Box.java
 *
 * Created on 17. 10. 2014, 11:28:11 by burgetr
 */
package org.fit.layout.model;

import java.awt.Color;

/**
 * This class represents a box in the rendered page tree. It may contain a text or a content
 * object (e.g. an image).
 * 
 * @author burgetr
 */
public interface Box extends ContentRect
{

    /**
     * Obtains the parent box of this box.
     * @return The parent box or {@code null} when this is the root box.
     */
    public Box getParentBox();
    
    /**
     * Obtains the n-th child box.
     * @param index the child index
     * @return the child box at the given index
     */
    public Box getChildBox(int index)
        throws ArrayIndexOutOfBoundsException;
    
    /**
     * Returns the number of child boxes for this box.
     * @return the number of child boxes
     */
    public int getChildCount();
    
    /**
     * Obtains the box text color.
     * @return the text color
     */
    public Color getColor();

    /**
     * Obtains the font family name used by the box.
     * @return The font family name.
     */
    public String getFontFamily();
    
    /**
     * Obtains the complete text contained in this box. 
     * @return a text string (possibly empty)
     */
    public String getText();
    
    /**
     * Obtains the content object contained in this box.
     * @return the content object or {@code null} when there is no object contained.
     */
    public ContentObject getContentObject();
    
    //=============================================================================================
    // BOUNDS
    //=============================================================================================
    
    /** 
     * Returns the logical bounds of the box node. Normally, the bounds are the same
     * as the content bounds. However, the BoxNode may be extended
     * in order to enclose all the overlapping boxes
     * @return the logical bounds of the box
     */
    public Rectangular getBounds();
    
    /**
     * Returns the content bounds. They correspond to the background bounds of the box;
     * however, when a border is present, it is included in the contents. Moreover,
     * the box is clipped by its clipping box.
     * @return the content bounds of the box
     */
    public Rectangular getContentBounds();

    /**
     * Returns the bounds of the box as they visually appear to the user.
     * @return the visual bounds
     */
    public Rectangular getVisualBounds();

    /**
     * Checks whether the box is visible.
     * @return {@code true} when the box is visible
     */
    public boolean isVisible();

    //=============================================================================================
    // BOX TYPE
    //=============================================================================================
    
    /**
     * Resurns the type of the box.
     * @return the box type
     */
    public Type getType();
    
    /**
     * The type of the box.
     * 
     * @author burgetr
     */
    public enum Type 
    {
        /** Boxes produced by a DOM element */
        ELEMENT,
        /** Text boxes (produced by DOM text node */
        TEXT_CONTENT,
        /** Replaced boxes (images, objects, etc.) */
        REPLACED_CONTENT
    }
    
    //=============================================================================================
    // DOM RELATIONSHIP
    //=============================================================================================
    
    /**
     * When the box is generated by a DOM element, obtains the corresponding tag name.
     * @return the tag name or {@code null} for boxes that do not correspond to a DOM element.
     */
    public String getTagName();

    /**
     * Obtains a value of an additional attribute. The existing attribute names depend on the box implementation;
     * they may correspond to HTML (DOM) attributes or they may be purely virtual. The typical attributes to be
     * implemented are {@code class} or {@code href}.
     * @param name the attribute name
     * @return the attribute value or {@code null} if the value is not specified.
     */
    public String getAttribute(String name);
    
    /**
     * Obtains the display type of the element according to the CSS display: property.
     * @return The display type of NULL if the box does not correspond to an element
     */
    public DisplayType getDisplayType();
    
    /**
     * CSS Display Type
     * 
     * @author burgetr
     */
    public enum DisplayType
    {
        NONE,         
        INLINE, BLOCK, LIST_ITEM, INLINE_BLOCK, RUN_IN,
        TABLE, INLINE_TABLE, TABLE_ROW_GROUP, TABLE_HEADER_GROUP, TABLE_FOOTER_GROUP,
        TABLE_ROW, TABLE_COLUMN_GROUP, TABLE_COLUMN, TABLE_CELL, TABLE_CAPTION
    }
    
    
}
