/**
 * SortByLinesOperator.java
 *
 * Created on 17. 9. 2015, 13:49:32 by burgetr
 */
package org.fit.segm.grouping.op;

import java.util.List;
import java.util.Vector;

import org.fit.layout.model.Area;
import org.fit.layout.model.AreaTree;
import org.fit.segm.grouping.AreaImpl;
import org.fit.segm.grouping.AreaUtils;

/**
 * 
 * @author burgetr
 */
public class SortByLinesOperator extends SortByPositionOperator
{
    protected final String[] paramNames = { };
    protected final ValueType[] paramTypes = { };
    
    public SortByLinesOperator()
    {
        super(false);
    }
    
    @Override
    public String getId()
    {
        return "FitLayout.Segm.SortByLines";
    }
    
    @Override
    public String getName()
    {
        return "Sort by lines";
    }

    @Override
    public String getDescription()
    {
        return "Sorts the visual areas roughly according to the text lines detected in the file";
    }

    @Override
    public String[] getParamNames()
    {
        return paramNames;
    }

    @Override
    public ValueType[] getParamTypes()
    {
        return paramTypes;
    }
    
    //==============================================================================

    @Override
    public void apply(AreaTree atree)
    {
        apply(atree, atree.getRoot());
    }

    @Override
    public void apply(AreaTree atree, Area root)
    {
        recursivelySortChildAreas(root, false);
        recursiveSortLines((AreaImpl) root);
    }
    
    //==============================================================================
    
    /**
     * Goes through all the areas in the tree and sorts their sub-areas.
     */
    protected void recursiveSortLines(AreaImpl root)
    {
        sortChildLines(root);
        for (int i = 0; i < root.getChildCount(); i++)
            recursiveSortLines((AreaImpl) root.getChildArea(i));
    }
    
    /**
     * Goes through the grid of areas and sorts the adjacent visual areas that are not
     * separated by anything
     */
    protected void sortChildLines(AreaImpl root)
    {
        if (root.getGrid() == null) //a gird is necessary for this
            root.createGrid();
        if (root.getChildCount() > 1)
        {
            List<Area> src = new Vector<Area>(root.getChildAreas());
            List<Area> dest = new Vector<Area>(src.size());
            while (!src.isEmpty())
            {
                final AreaImpl seed = (AreaImpl) src.get(0);
                List<Area> line = findAreasOnLine(root, seed, src);
                dest.addAll(line);
                src.removeAll(line);
            }
            
            root.removeAllChildren();
            root.appendChildren(dest);
        }
    }

    private List<Area> findAreasOnLine(AreaImpl parent, AreaImpl area, List<Area> candidates)
    {
        Vector<Area> ret = new Vector<Area>();
        ret.add(area);
        
        final int nx1 = area.getGridPosition().getX1();
        final int ny1 = area.getGridPosition().getY1();
        final int nx2 = area.getGridPosition().getX2();
        final int ny2 = area.getGridPosition().getY2();
        
        //try to expand to the right
        int dist = 1;
        while (nx2 + dist < parent.getGrid().getWidth())
        {
            //try to find some node at the right in the given distance
            for (int y = ny1; y <= ny2; y++)
            {
                AreaImpl neigh = (AreaImpl) parent.getGrid().getAreaAt(nx2 + dist, y);
                if (neigh != null && candidates.contains(neigh)) //something found
                {
                    //the maximal Y difference to consider other areas to be on the same line
                    int threshold = (Math.min(area.getHeight(), neigh.getHeight()) / 2);
                    if (threshold < 0) threshold = 0;
                    //check if the nodes could be on the same line
                    if (AreaUtils.isOnSameLine(area, neigh, threshold))
                    {
                        ret.add(neigh);
                        break;
                    }
                }
            }
            dist++;
        }
        //try to expand to the left
        dist = 1;
        while (nx1 - dist >= 0)
        {
            //try to find some node at the right in the given distance
            for (int y = ny1; y <= ny2; y++)
            {
                AreaImpl neigh = (AreaImpl) parent.getGrid().getAreaAt(nx1 - dist, y);
                if (neigh != null && candidates.contains(neigh)) //something found
                {
                    //the maximal Y difference to consider other areas to be on the same line
                    int threshold = (Math.min(area.getHeight(), neigh.getHeight()) / 2);
                    if (threshold < 0) threshold = 0;
                    //check if the nodes could be on the same line
                    if (AreaUtils.isOnSameLine(area, neigh, threshold))
                    {
                        ret.insertElementAt(neigh, 0);
                        break;
                    }
                }
            }
            dist++;
        }
            
        return ret;
    }
    
}
