/**
 * XMLOutputOperator.java
 *
 * Created on 2. 2. 2015, 13:04:11 by burgetr
 */
package org.fit.layout.tools.io;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.Set;
import java.util.Vector;

import org.fit.layout.impl.BaseOperator;
import org.fit.layout.model.Area;
import org.fit.layout.model.AreaTree;
import org.fit.layout.model.Box;
import org.fit.layout.model.Box.Type;
import org.fit.layout.model.Rectangular;
import org.fit.layout.model.Tag;

/**
 * This operator serializes the area tree to an XML file.
 *  
 * @author burgetr
 */
public class XMLOutputOperator extends BaseOperator
{
    /** Should we produce the XML header? */
    protected boolean produceHeader;
    
    /** Path to the output file/ */
    protected String filename;
    
    protected final String[] paramNames = { "filename", "produceHeader" };
    protected final ValueType[] paramTypes = { ValueType.STRING, ValueType.BOOLEAN };

    private int idcnt = 0;

    
    public XMLOutputOperator()
    {
        produceHeader = false;
        filename = "out.xml";
    }

    public XMLOutputOperator(String filename, boolean produceHeader)
    {
        this.filename = filename;
        this.produceHeader = produceHeader;
    }

    @Override
    public String getId()
    {
        return "FitLayout.Tools.XMLOutput";
    }

    @Override
    public String getName()
    {
        return "XML serialization of the area tree";
    }

    @Override
    public String getDescription()
    {
        return "Serializes the area tree to an XML file";
    }

    @Override
    public String[] getParamNames()
    {
        return paramNames;
    }

    @Override
    public ValueType[] getParamTypes()
    {
        return paramTypes;
    }

    public boolean getProduceHeader()
    {
        return produceHeader;
    }

    public void setProduceHeader(boolean produceHeader)
    {
        this.produceHeader = produceHeader;
    }

    public String getFilename()
    {
        return filename;
    }

    public void setFilename(String filename)
    {
        this.filename = filename;
    }

    //=====================================================================================================
    
    @Override
    public void apply(AreaTree atree)
    {
        apply(atree, atree.getRoot());
    }

    @Override
    public void apply(AreaTree atree, Area root)
    {
        try
        {
            PrintWriter out = new PrintWriter(filename);
            dumpTo(atree, out);
            out.close();
        } catch (FileNotFoundException e) {
            System.err.println("Couldn't create output XML file " + filename);
        }
    }

    //=====================================================================================================
    
    /**
     * Formats the complete tag tree to an output stream
     */
    public void dumpTo(AreaTree tree, PrintWriter out)
    {
        if (produceHeader)
            out.println("<?xml version=\"1.0\"?>");
        out.println("<areaTree base=\"" + HTMLEntities(tree.getRoot().getPage().getSourceURL().toString()) + "\">");
        recursiveDump(tree.getRoot(), 1, out);
        out.println("</areaTree>");
    }
    
    //=====================================================================
    
    private void recursiveDump(Area a, int level, java.io.PrintWriter p)
    {
        String stag = "<area"
                        + " id=\"x" + (idcnt++) + "\""
                        + " x1=\"" + a.getX1() + "\"" 
                        + " y1=\"" + a.getY1() + "\"" 
                        + " x2=\"" + a.getX2() + "\"" 
                        + " y2=\"" + a.getY2() + "\"" 
                        + " background=\"" + colorString(a.getBackgroundColor()) + "\"" 
                        + " fontsize=\"" + a.getFontSize() + "\"" 
                        + " fontweight=\"" + a.getFontWeight() + "\"" 
                        + " fontstyle=\"" + a.getFontStyle() + "\""
                        + " tags=\"" + tagString(a.getTags().keySet()) + "\""
                        + ">";

        String etag = "</area>";
        
        if (a.getChildCount() > 0)
        {
            indent(level, p);
            p.println(stag);
            
            for (int i = 0; i < a.getChildCount(); i++)
                recursiveDump(a.getChildArea(i), level+1, p);
            
            indent(level, p);
            p.println(etag);
        }
        else
        {
            indent(level, p);
            p.println(stag);
            dumpBoxes(a, p, level+1);
            indent(level, p);
            p.println(etag);
        }
        
    }
    
    private void dumpBoxes(Area a, java.io.PrintWriter p, int level)
    {
        Vector<Box> boxes = a.getBoxes();
        for (Box box : boxes)
        {
            Rectangular pos = box.getVisualBounds();
            indent(level, p);
            String stag = "<box"
                            + " x1=\"" + pos.getX1() + "\"" 
                            + " y1=\"" + pos.getY1() + "\"" 
                            + " x2=\"" + pos.getX2() + "\"" 
                            + " y2=\"" + pos.getY2() + "\""
                            + " color=\"" + colorString(box.getColor()) + "\""
                            + " fontfamily=\"" + box.getFontFamily() + "\""
                            + " fontsize=\"" + box.getFontSize() + "\""
                            + " fontweight=\"" + (box.getFontWeight()) + "\""
                            + " fontstyle=\"" + (box.getFontStyle()) + "\""
                            + " underline=\"" + box.getUnderline() + "\""
                            + " linethrough=\"" + box.getLineThrough() + "\""
                            + " replaced=\"" + ((box.getType() == Type.REPLACED_CONTENT)?"true":"false") + "\""
                            + ">";
            p.print(stag);
            p.print(HTMLEntities(box.getText()));
            p.println("</box>");
        }
    }
    
    private void indent(int level, java.io.PrintWriter p)
    {
        String ind = "";
        for (int i = 0; i < level*4; i++) ind = ind + ' ';
        p.print(ind);
    }
    
    private String colorString(java.awt.Color color)
    {
        if (color == null)
            return "";
        else
            return String.format("#%02x%02x%02x", color.getRed(), color.getGreen(), color.getBlue());
    }
    
    /**
     * Converts the CSS specification rgb(r,g,b) to #rrggbb
     * @param spec the CSS color specification
     * @return a #rrggbb string
     */
    public String colorString(String spec)
    {
        if (spec.startsWith("rgb("))
        {
            String s = spec.substring(4, spec.length() - 1);
            String[] lst = s.split(",");
            try {
                int r = Integer.parseInt(lst[0].trim());
                int g = Integer.parseInt(lst[1].trim());
                int b = Integer.parseInt(lst[2].trim());
                return String.format("#%02x%02x%02x", r, g, b);
            } catch (NumberFormatException e) {
                return null;
            }
        }
        else
            return spec;
    }
    
    private String tagString(Set<Tag> tags)
    {
        String ret = "";
        for (Tag tag : tags)
            ret += tag + " ";
        return ret.trim();
    }
    
    private String HTMLEntities(String s)
    {
        return s.replaceAll(">", "&gt;").replaceAll("<", "&lt;").replaceAll("&", "&amp;");
    }
    
}
