/*
 * Decompiled with CFR 0.152.
 */
package cz.xtf.core.helm;

import cz.xtf.core.config.HelmConfig;
import cz.xtf.core.helm.HelmBinaryPathResolver;
import cz.xtf.core.http.Https;
import cz.xtf.core.openshift.ClusterVersionInfo;
import cz.xtf.core.openshift.ClusterVersionInfoFactory;
import cz.xtf.core.utils.CoreUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConfiguredVersionHelmBinaryPathResolver
implements HelmBinaryPathResolver {
    private static final Logger log = LoggerFactory.getLogger(ConfiguredVersionHelmBinaryPathResolver.class);
    private static final String OCP_4_HELM_BINARY_DOWNLOAD_URL = "https://mirror.openshift.com/pub/openshift-v4/clients/helm";

    ConfiguredVersionHelmBinaryPathResolver() {
    }

    @Override
    public String resolve() {
        ClusterVersionInfo clusterVersionInfo = ClusterVersionInfoFactory.INSTANCE.getClusterVersionInfo();
        if (!clusterVersionInfo.getOpenshiftVersion().startsWith("4")) {
            log.warn("Unsupported Openshift version for Helm client, OCP cluster is of version {}, while currently only OCP 4 is supported", (Object)clusterVersionInfo.getOpenshiftVersion());
            return null;
        }
        boolean cacheEnabled = HelmConfig.isHelmBinaryCacheEnabled();
        String helmClientVersion = HelmConfig.helmClientVersion();
        String clientUrl = this.getHelmClientUrlBasedOnConfiguredHelmVersion(helmClientVersion);
        Path archivePath = this.getCachedOrDownloadClientArchive(clientUrl, helmClientVersion, cacheEnabled);
        return this.unpackHelmClientArchive(archivePath, !cacheEnabled);
    }

    private String unpackHelmClientArchive(Path archivePath, boolean deleteArchiveWhenDone) {
        Objects.requireNonNull(archivePath);
        try {
            List<String> args = Stream.of("tar", "-xf", archivePath.toAbsolutePath().toString(), "-C", this.getProjectHelmDir().toAbsolutePath().toString(), "--no-same-owner").collect(Collectors.toList());
            ProcessBuilder pb = new ProcessBuilder(args);
            pb.redirectOutput(ProcessBuilder.Redirect.INHERIT);
            pb.redirectError(ProcessBuilder.Redirect.INHERIT);
            int result = pb.start().waitFor();
            if (result != 0) {
                throw new IOException("Failed to execute: " + args);
            }
        }
        catch (IOException | InterruptedException e) {
            throw new IllegalStateException("Failed to extract helm binary " + archivePath.toAbsolutePath(), e);
        }
        try {
            if (deleteArchiveWhenDone) {
                Files.delete(archivePath);
            }
        }
        catch (IOException ioe) {
            log.warn("It wasn't possible to delete Helm client archive {}", (Object)archivePath.toAbsolutePath(), (Object)ioe);
        }
        try (Stream<Path> helmDirContents = Files.list(this.getProjectHelmDir());){
            Path helmBinaryFile = (Path)helmDirContents.collect(Collectors.toList()).get(0);
            if (!helmBinaryFile.getFileName().toString().equals("helm")) {
                Files.move(helmBinaryFile, helmBinaryFile.resolveSibling("helm"), StandardCopyOption.REPLACE_EXISTING);
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Error when extracting Helm client binary", e);
        }
        return this.getProjectHelmDir().resolve("helm").toAbsolutePath().toString();
    }

    private Path getCachedOrDownloadClientArchive(String url, String version, boolean cacheEnabled) {
        Path archivePath;
        Objects.requireNonNull(url);
        log.debug("Trying to load Helm client archive from cache (enabled: {}) or download it from {}.", (Object)cacheEnabled, (Object)url);
        if (cacheEnabled) {
            Path cachePath = Paths.get(HelmConfig.binaryCachePath(), version, DigestUtils.md5Hex((String)url));
            archivePath = cachePath.resolve("helm.tar.gz");
            if (Files.exists(archivePath, new LinkOption[0])) {
                log.debug("Helm client archive is already in cache: {}.", (Object)archivePath.toAbsolutePath());
                return archivePath;
            }
            log.debug("Helm client archive not found in cache, downloading it.");
        } else {
            archivePath = this.getProjectHelmDir().resolve("helm.tar.gz");
            log.debug("Cache is disabled, downloading Helm client archive to {}.", (Object)archivePath.toAbsolutePath());
        }
        try {
            Https.copyHttpsURLToFile(url, archivePath.toFile(), 20000, 300000);
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Failed to download and extract helm binary from " + url, ioe);
        }
        return archivePath;
    }

    private String getHelmClientUrlBasedOnConfiguredHelmVersion(String helmClientVersion) {
        String systemType = SystemUtils.IS_OS_MAC ? "darwin" : "linux";
        String sysArch = CoreUtils.getSystemArchForOCP4();
        return String.format("%s/%s/helm-%s-%s.tar.gz", OCP_4_HELM_BINARY_DOWNLOAD_URL, helmClientVersion, systemType, sysArch);
    }

    private Path getProjectHelmDir() {
        Path dir = Paths.get("tmp/helm/", new String[0]);
        try {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Failed to create directory " + dir.toAbsolutePath(), ioe);
        }
        return dir;
    }
}

