/*
 * Decompiled with CFR 0.152.
 */
package cz.xtf.core.openshift;

import cz.xtf.core.config.OpenShiftConfig;
import cz.xtf.core.http.Https;
import cz.xtf.core.openshift.ClusterVersionInfo;
import cz.xtf.core.openshift.ClusterVersionInfoFactory;
import cz.xtf.core.openshift.OpenShiftBinaryPathResolver;
import cz.xtf.core.openshift.OpenShifts;
import cz.xtf.core.utils.CoreUtils;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.openshift.api.model.Route;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ClusterVersionOpenShiftBinaryPathResolver
implements OpenShiftBinaryPathResolver {
    private static final Logger log = LoggerFactory.getLogger(ClusterVersionOpenShiftBinaryPathResolver.class);
    private static final String OCP3_CLIENTS_URL = "https://mirror.openshift.com/pub/openshift-v3/clients";
    private static final String OCP4_CLIENTS_URL = "https://mirror.openshift.com/pub/openshift-v4";

    ClusterVersionOpenShiftBinaryPathResolver() {
    }

    @Override
    public String resolve() {
        ClusterVersionInfo clusterVersionInfo = ClusterVersionInfoFactory.INSTANCE.getClusterVersionInfo();
        boolean cacheEnabled = OpenShiftConfig.isBinaryCacheEnabled();
        String clientUrl = this.determineClientUrl(clusterVersionInfo);
        Path archivePath = this.getCachedOrDownloadClientArchive(clientUrl, clusterVersionInfo.getOpenshiftVersion() != null ? this.getVersionOrChannel(clusterVersionInfo) : "unknown", cacheEnabled);
        return this.unpackOpenShiftClientArchive(archivePath, !cacheEnabled);
    }

    private String determineClientUrl(ClusterVersionInfo versionInfo) {
        log.debug("Trying to determine OpenShift client url for cluster version {}.", (Object)versionInfo.getOpenshiftVersion());
        if (versionInfo.getOpenshiftVersion() != null) {
            return this.getClientUrlBasedOnOcpVersion(versionInfo);
        }
        return this.loadOrGuessClientUrlOnCluster();
    }

    private String getClientUrlBasedOnOcpVersion(ClusterVersionInfo versionInfo) {
        Objects.requireNonNull(versionInfo);
        String openshiftVersion = versionInfo.getOpenshiftVersion();
        if (openshiftVersion.startsWith("3")) {
            String systemTypeForOCP3 = CoreUtils.getSystemTypeForOCP3();
            String downloadUrl = String.format("%s/%s/%s/oc.tar.gz", OCP3_CLIENTS_URL, openshiftVersion, systemTypeForOCP3);
            int code = Https.httpsGetCode(downloadUrl);
            if (code >= 400 && code < 500) {
                downloadUrl = String.format("%s/%s-1/%s/oc.tar.gz", OCP3_CLIENTS_URL, openshiftVersion, systemTypeForOCP3);
            }
            return downloadUrl;
        }
        String downloadUrl = this.getOcp4DownloadUrl(versionInfo, this.getVersionOrChannel(versionInfo));
        int code = Https.httpsGetCode(downloadUrl);
        if (code >= 400 && code < 500) {
            downloadUrl = this.getOcp4DownloadUrl(versionInfo, this.getConfiguredChannel());
        }
        return downloadUrl;
    }

    private String getOcp4DownloadUrl(ClusterVersionInfo versionInfo, String versionOrChannel) {
        String ocFileName;
        String string = ocFileName = SystemUtils.IS_OS_MAC ? "openshift-client-mac.tar.gz" : "openshift-client-linux.tar.gz";
        if (versionInfo.isDeveloperPreview()) {
            log.warn("Developer preview version detected, defaulting to the latest stable channel for oc binary download.");
            versionOrChannel = "stable";
        }
        return String.format("%s/%s/clients/ocp/%s/%s", OCP4_CLIENTS_URL, CoreUtils.getSystemArchForOCP4(), versionOrChannel, ocFileName);
    }

    private String loadOrGuessClientUrlOnCluster() {
        String locationTemplate = "https://%s/%s/%s/oc.tar";
        String systemArch = CoreUtils.getSystemArchForOCP4();
        String operatingSystem = SystemUtils.IS_OS_MAC ? "mac" : "linux";
        try {
            Optional<Route> downloadsRouteOptional = Optional.ofNullable(OpenShifts.admin("openshift-console").getRoute("downloads"));
            Route downloads = downloadsRouteOptional.orElseThrow(() -> new IllegalStateException("We are not able to find download link for OC binary."));
            return String.format("https://%s/%s/%s/oc.tar", downloads.getSpec().getHost(), systemArch, operatingSystem);
        }
        catch (KubernetesClientException kce) {
            log.warn("It isn't possible to read 'downloads' route in 'openshift-console' namespace to get binary location. Attempting to guess it.", (Throwable)kce);
            return String.format("https://%s/%s/%s/oc.tar", OpenShifts.admin("openshift-console").generateHostname("downloads"), systemArch, operatingSystem);
        }
    }

    private Path getCachedOrDownloadClientArchive(String url, String version, boolean cacheEnabled) {
        Path archivePath;
        Objects.requireNonNull(url);
        log.debug("Trying to load OpenShift client archive from cache (enabled: {}) or download it from {}.", (Object)cacheEnabled, (Object)url);
        if (cacheEnabled) {
            Path cachePath = Paths.get(OpenShiftConfig.binaryCachePath(), version, DigestUtils.md5Hex((String)url));
            archivePath = cachePath.resolve("oc.tar.gz");
            if (Files.exists(archivePath, new LinkOption[0])) {
                log.debug("OpenShift client archive is already in cache: {}.", (Object)archivePath.toAbsolutePath());
                return archivePath;
            }
            log.debug("OpenShift client archive not found in cache, downloading it.");
        } else {
            archivePath = this.getProjectOcDir().resolve("oc.tar.gz");
            log.debug("Cache is disabled, downloading OpenShift client archive to {}.", (Object)archivePath.toAbsolutePath());
        }
        try {
            Https.copyHttpsURLToFile(url, archivePath.toFile(), 20000, 300000);
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Failed to download and extract oc binary from " + url, ioe);
        }
        return archivePath;
    }

    private String unpackOpenShiftClientArchive(Path archivePath, boolean deleteArchiveWhenDone) {
        Objects.requireNonNull(archivePath);
        try {
            List<String> args = Stream.of("tar", "-xf", archivePath.toAbsolutePath().toString(), "-C", this.getProjectOcDir().toAbsolutePath().toString()).collect(Collectors.toList());
            ProcessBuilder pb = new ProcessBuilder(args);
            pb.redirectOutput(ProcessBuilder.Redirect.INHERIT);
            pb.redirectError(ProcessBuilder.Redirect.INHERIT);
            int result = pb.start().waitFor();
            if (result != 0) {
                throw new IOException("Failed to execute: " + args);
            }
        }
        catch (IOException | InterruptedException e) {
            throw new IllegalStateException("Failed to extract oc binary " + archivePath.toAbsolutePath(), e);
        }
        try {
            if (deleteArchiveWhenDone) {
                Files.delete(archivePath);
            }
        }
        catch (IOException ioe) {
            log.warn("It wasn't possible to delete OpenShift client archive {}", (Object)archivePath.toAbsolutePath(), (Object)ioe);
        }
        return this.getProjectOcDir().resolve("oc").toAbsolutePath().toString();
    }

    private Path getProjectOcDir() {
        Path dir = Paths.get("tmp/oc/", new String[0]);
        try {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Failed to create directory " + dir.toAbsolutePath(), ioe);
        }
        return dir;
    }

    private String getVersionOrChannel(ClusterVersionInfo versionInfo) {
        Objects.requireNonNull(versionInfo);
        if (versionInfo.isMajorMinorMicro()) {
            return versionInfo.getOpenshiftVersion();
        }
        return this.getConfiguredChannel() + "-" + versionInfo.getMajorMinorOpenshiftVersion();
    }

    private String getConfiguredChannel() {
        String channel = OpenShiftConfig.binaryUrlChannelPath();
        if (!Stream.of("stable", "fast", "latest", "candidate").collect(Collectors.toList()).contains(channel)) {
            throw new IllegalStateException("Channel (" + channel + ") configured in 'xtf.openshift.binary.url.channel' property is invalid.");
        }
        return channel;
    }
}

