/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.alpharogroup.collections;

import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * This class overwrites the put-method from the <tt>LinkedHashSet</tt>. That inserts the value to
 * the right order it was inserted in the Map. Note that the difference in the LinkedHashMap is the
 * order does not change if we put the same key with a new value. In this class the order changes
 * when we put a new value with the same key.
 * 
 * @author Asterios Raptis
 * @version 1.0
 * @param <E>
 *            the type of elements maintained by this set
 * @see java.util.LinkedHashSet
 */
public class InsertionOrderSet<E> extends LinkedHashSet<E>
{

	/**
	 * The serialVersionUID.
	 */
	private static final long serialVersionUID = -2010420135340052455L;

	/**
	 * Constructs an empty insertion-ordered <tt>InsertionOrderSet</tt> instance with a default
	 * capacity (16) and load factor (0.75).
	 */
	public InsertionOrderSet()
	{
		super();
	}

	/**
	 * Constructs a new insertion order hash set with the same elements as the specified collection.
	 * The insertion order hash set is created with an initial capacity sufficient to hold the
	 * elements in the specified collection and the default load factor (0.75).
	 * 
	 * @param c
	 *            the collection whose elements are to be placed into this set.
	 * @throws NullPointerException
	 *             if the specified collection is null.
	 */
	public InsertionOrderSet(final Collection<? extends E> c)
	{
		super(c);
	}

	/**
	 * Constructs an empty insertion-ordered <tt>InsertionOrderSet</tt> instance with the specified
	 * initial capacity and a default load factor (0.75).
	 * 
	 * @param initialCapacity
	 *            the initial capacity.
	 * @throws IllegalArgumentException
	 *             if the initial capacity is negative.
	 */
	public InsertionOrderSet(final int initialCapacity)
	{
		super(initialCapacity);
	}

	/**
	 * Constructs an empty insertion-ordered <tt>InsertionOrderMap</tt> instance with the specified
	 * initial capacity and load factor.
	 * 
	 * @param initialCapacity
	 *            the initial capacity.
	 * @param loadFactor
	 *            the load factor.
	 * @throws IllegalArgumentException
	 *             if the initial capacity is negative or the load factor is nonpositive.
	 */
	public InsertionOrderSet(final int initialCapacity, final float loadFactor)
	{
		super(initialCapacity, loadFactor);
	}

	/**
	 * Adds the specified object to this set. If the set previously contained a the same object, the
	 * old object is removed and then insert again. That inserts the object to the right order it
	 * was inserted in the Set. Note that the difference in the LinkedHashSet is the order does not
	 * change if we try to add the same object. In this class the order changes when we add a object
	 * that allready exists.
	 * 
	 * @param o
	 *            The object to add.
	 * @return true if the set did not already contain the specified element.
	 */
	@Override
	public boolean add(final E o)
	{
		if (super.contains(o))
		{
			super.remove(o);
		}
		return super.add(o);
	}

	/**
	 * Factory method for create an InsertionOrderSet with the given elements.
	 * 
	 * @param <E>
	 *            the generic type of the elements
	 * @param elements
	 *            The given elements.
	 * @return a new InsertionOrderSet that contains the given elements.
	 */
	@SuppressWarnings("unchecked")
	public static <E> Set<E> setOf(E... elements)
	{
		return new InsertionOrderSet<>(Arrays.asList(elements));
	}

}
