/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.collections.modifications;

import java.util.Collection;

/**
 * The Enum ModifiedState represents states if a collection have been modified.
 * Checks the same( or different) collection in different states. If a reference of a
 * collection in the past is same or modified with the reference of the
 * collection now.
 */
public enum ModifiedState {

	/**
	 * The cleared state. This indicates that no entry exists anymore in the
	 * collection.
	 **/
	CLEARED,

	/**
	 * The first match state. This indicates that there is a total new match.
	 * Precondition is that no entries exists in the collection.
	 **/
	FIRST_MATCH,

	/**
	 * The new match state. This indicates that an new entry is added.
	 * Precondition is that entries exists in the collection.
	 **/
	NEW_MATCH,

	/**
	 * The removed state. This indicates that an entry has been removed.
	 * Precondition is that minimun an entry exists in the collection.
	 * */
	REMOVED,

	/**
	 * The unmodified state. This indicates that no entry added or removed from
	 * the collection.
	 **/
	UNMODIFIED;

	/**
	 * Checks if the given previous(for instance in the past) Collection is modified comparing the next(for instance now) Collection.
	 *
	 * @param <T>
	 *            the type.
	 * @param previous
	 *            the previous collection.
	 * @param next
	 *            the next collection.
	 * @return the modified
	 */
	public static <T> ModifiedState isModified(Collection<T> previous,
			Collection<T> next) {
		if (previous == null) {
			throw new IllegalArgumentException(
					"First argument must not be null.");
		}
		if (next == null) {
			throw new IllegalArgumentException(
					"Second argument must not be null.");
		}

		boolean collectionsEqual = previous.containsAll(next)
				&& next.containsAll(previous);

		if (collectionsEqual) {
			return ModifiedState.UNMODIFIED;
		}

		boolean cleared = next.isEmpty() && !previous.isEmpty();

		if (cleared) {
			return ModifiedState.CLEARED;
		}
		boolean removed = next.size() < previous.size();

		if (removed) {
			return ModifiedState.REMOVED;
		}
		boolean initialEntry = previous.isEmpty() && !next.isEmpty();

		if (initialEntry) {
			return ModifiedState.FIRST_MATCH;
		}

		boolean bothCollectionNotEmpty = !next.isEmpty() && !previous.isEmpty();
		boolean nextGreaterThanPrevious = previous.size() < next.size();
		boolean newMatch = bothCollectionNotEmpty && !collectionsEqual
				|| nextGreaterThanPrevious;

		if (newMatch) {
			return ModifiedState.NEW_MATCH;
		}

		return ModifiedState.UNMODIFIED;
	}

}
