/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.email.messages;

import java.io.File;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;

import lombok.Getter;
import net.sourceforge.jaulp.email.data.sources.ByteArrayDataSource;

/**
 * This class adds the functionality to add attachments to the EmailMessage.
 *
 * @author Asterios Raptis
 */
public class EmailMessageWithAttachments
{

	/** The EmailMessage. */
	@Getter
	private final EmailMessage emailMessage;

	/** The MimeMultipart for the attachments. */
	private final MimeMultipart mimeMultiPart;

	/**
	 * Constructor for the EmailAttachments.
	 *
	 * @param emailMessage
	 *            The EmailMessage.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public EmailMessageWithAttachments(final EmailMessage emailMessage) throws MessagingException
	{
		mimeMultiPart = new MimeMultipart();
		this.emailMessage = emailMessage;
		this.emailMessage.setContent(mimeMultiPart);
	}

	/**
	 * Adds an attachment to the EmailMessage.
	 *
	 * @param part
	 *            The BodyPart to add to the EmailMessage.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public void addAttachment(final BodyPart part) throws MessagingException
	{
		getMimeMultiPart().addBodyPart(part);
	}

	/**
	 * Adds the attachment.
	 *
	 * @param content
	 *            The bytearray with the content.
	 * @param filename
	 *            The new Filename for the attachment.
	 * @param mimetype
	 *            The mimetype.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public void addAttachment(final byte[] content, final String filename, final String mimetype)
		throws MessagingException
	{
		final DataSource dataSource = new ByteArrayDataSource(content, mimetype);
		final DataHandler dataHandler = new DataHandler(dataSource);
		addAttachment(dataHandler, filename);
	}

	/**
	 * Adds the attachment.
	 *
	 * @param handler
	 *            The DataHandler for the attachment.
	 * @param filename
	 *            The new Filename for the attachment.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public void addAttachment(final DataHandler handler, final String filename)
		throws MessagingException
	{
		final BodyPart part = new MimeBodyPart();
		part.setFileName(filename);
		part.setDataHandler(handler);
		addAttachment(part);
	}

	/**
	 * Adds an attachment to the EmailMessage.
	 *
	 * @param file
	 *            The file to add to the EmailMessage.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public void addAttachment(final File file) throws MessagingException
	{
		DataSource dataSource;
		dataSource = new FileDataSource(file);
		final DataHandler dataHandler = new DataHandler(dataSource);
		addAttachment(dataHandler, file.getName());
	}

	/**
	 * Adds an attachment to the EmailMessage.
	 *
	 * @param file
	 *            The file to add to the EmailMessage.
	 * @param newFilename
	 *            The new Filename for the attachment.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public void addAttachment(final File file, final String newFilename) throws MessagingException
	{
		DataSource dataSource;
		dataSource = new FileDataSource(file);
		final DataHandler dataHandler = new DataHandler(dataSource);
		addAttachment(dataHandler, newFilename);
	}

	/**
	 * Gets the MimeMultipart from the EmailAttachments.
	 *
	 * @return 's the MimeMultipart from the EmailAttachments.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public synchronized MimeMultipart getMimeMultiPart() throws MessagingException
	{
		return mimeMultiPart;
	}

}
