/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.email.messages;

import java.io.File;
import java.net.FileNameMap;
import java.net.URLConnection;

import eu.medsea.mimeutil.MimeUtil2;

/**
 * The Enum Mimetypes.
 */
public enum Mimetypes
{

	/** The image jpeg. */
	IMAGE_JPEG("image/jpeg"),

	/** The image pjpeg. */
	IMAGE_PJPEG("image/pjpeg"),

	/** The image png. */
	IMAGE_PNG("image/png"),

	/** The image gif. */
	IMAGE_GIF("image/gif"),

	/** The image tiff. */
	IMAGE_TIFF("image/tiff"),

	/** The image ief. */
	IMAGE_IEF("image/ief"),

	/** The image x icon. */
	IMAGE_X_ICON("image/x-icon"),

	/** The image x xcf.GIMP image file */
	IMAGE_X_XCF("image/x-xcf"),

	/** The text csv. */
	TEXT_CSV("text/csv"),

	/** The eml. */
	EML("message/rfc822"),

	/** The text ics. */
	TEXT_ICS("text/calendar"),

	/** The text plain. */
	TEXT_PLAIN("text/plain"),

	/** The text html. */
	TEXT_HTML("text/html"),

	/** The text xml. */
	TEXT_XML("text/xml"),

	/** The application pdf. */
	APPLICATION_PDF("application/pdf"),

	/** The application rtf. */
	APPLICATION_RTF("application/rtf"),

	/** The application octet. */
	APPLICATION_OCTET("application/octet-stream");

	/** The mimetype. */
	private final String mimetype;

	/**
	 * Instantiates a new mimetypes.
	 *
	 * @param mimetype
	 *            the mimetype
	 */
	private Mimetypes(final String mimetype)
	{
		this.mimetype = mimetype;
	}

	/**
	 * Gets the mimetype.
	 *
	 * @return the mimetype
	 */
	public String getMimetype()
	{
		return mimetype;
	}

	/**
	 * Gets the mime type from the given File. It make use of the library mime-util
	 * 'http://sourceforge.net/projects/mime-util/'.
	 *
	 * @param file
	 *            the file
	 * @return the mime type
	 */
	public static String getMimeType(File file)
	{
		FileNameMap fileNameMap = URLConnection.getFileNameMap();
		String mimeType = fileNameMap.getContentTypeFor(file.getName());
		if (mimeType == null)
		{
			MimeUtil2 mimeUtil = new MimeUtil2();
			mimeUtil.registerMimeDetector("eu.medsea.mimeutil.detector.MagicMimeMimeDetector");
			mimeType = MimeUtil2.getMostSpecificMimeType(mimeUtil.getMimeTypes(file)).toString();
		}
		return mimeType;
	}

	/**
	 * Gets the mime type from the given File. It make use of the library mime-util
	 * 'http://sourceforge.net/projects/mime-util/'.
	 *
	 * @param file
	 *            the file
	 * @return the mime type
	 */
	public static String getExtension(File file)
	{
		String extension;
		MimeUtil2 mimeUtil = new MimeUtil2();
		mimeUtil.registerMimeDetector("eu.medsea.mimeutil.detector.MagicMimeMimeDetector");
		extension = MimeUtil2.getExtension(file);
		return extension;
	}

	/**
	 * Gets the mime type from the given File. It make use of the library mime-util
	 * 'http://sourceforge.net/projects/mime-util/'.
	 *
	 * @param fileName
	 *            the file name
	 * @return the extension
	 */
	public static String getExtension(final String fileName)
	{
		String extension;
		MimeUtil2 mimeUtil = new MimeUtil2();
		mimeUtil.registerMimeDetector("eu.medsea.mimeutil.detector.MagicMimeMimeDetector");
		extension = MimeUtil2.getExtension(fileName);
		return extension;
	}
}
