/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.date;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * Utility class for the use of Date and Calendar object.
 *
 * @author Asterios Raptis
 * @version 1.0
 */
public final class ConvertDateUtils implements DatePatterns
{

	/**
	 * Converts a Date to a Calendar-object.
	 *
	 * @param date
	 *            The date to convert.
	 * @return The calendar from the date.
	 */
	public static Calendar toCalendar(final Date date)
	{
		final Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		return calendar;
	}

	/**
	 * Converts the given long value to a calendar object.
	 *
	 * @param millis
	 *            the millis
	 * @return the calendar
	 */
	public static Calendar toCalendar(long millis)
	{
		final Calendar calendar = Calendar.getInstance();
		calendar.setTimeInMillis(millis);
		return calendar;
	}

	/**
	 * Converts a Calendar-object to a Date.
	 *
	 * @param calendar
	 *            The calendar to convert.
	 * @return The date from the calendar.
	 */
	public static Date toDate(final Calendar calendar)
	{
		final Date date = calendar.getTime();
		return date;
	}

	/**
	 * Converts the given long value to a Date object.
	 *
	 * @param millis
	 *            the millis
	 * @return the date
	 */
	public static Date toDate(final long millis)
	{
		return toDate(toCalendar(millis));
	}

	/**
	 * Converts a Date to a Timestamp-object.
	 *
	 * @param date
	 *            The date to convert.
	 * @return The Timestamp from the date.
	 */
	public static Timestamp toTimestamp(Date date)
	{
		Calendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);

		return new Timestamp(cal.getTime().getTime());
	}

	/**
	 * Util-class. Do not make instances from this class.
	 */
	private ConvertDateUtils()
	{
	}

}
