/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.date;

import java.security.SecureRandom;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;

/**
 * Utility class for the use of Date and Calendar object.
 *
 * @author Asterios Raptis
 * @version 1.0
 */
public final class CreateDateUtils implements DatePatterns
{

	/**
	 * Creates a new Date object from the given values.
	 *
	 * @param year
	 *            The year.
	 * @param month
	 *            The month.
	 * @param day
	 *            The day.
	 * @return Returns the created Date object.
	 */
	public static Date newDate(final int year, final int month, final int day)
	{
		return newDate(year, month, day, 0, 0, 0);
	}

	/**
	 * Creates a new Date object from the given values.
	 *
	 * @param year
	 *            The year.
	 * @param month
	 *            The month.
	 * @param day
	 *            The day.
	 * @param hour
	 *            The hour.
	 * @param min
	 *            The minute.
	 * @param sec
	 *            The second.
	 * @return Returns the created Date object.
	 */
	public static Date newDate(final int year, final int month, final int day, final int hour,
		final int min, final int sec)
	{
		return newDate(year, month, day, hour, min, sec, 0);
	}

	/**
	 * Creates a new Date object from the given values.
	 *
	 * @param year
	 *            The year.
	 * @param month
	 *            The month.
	 * @param day
	 *            The day.
	 * @param hour
	 *            The hour.
	 * @param minute
	 *            The minute.
	 * @param seconds
	 *            The second.
	 * @param milliSecond
	 *            The millisecond.
	 * @return Returns the created Date object.
	 */
	public static Date newDate(final int year, final int month, final int day, final int hour,
		final int minute, final int seconds, final int milliSecond)
	{
		final Calendar calendar = Calendar.getInstance();

		calendar.set(Calendar.YEAR, year);
		calendar.set(Calendar.MONTH, month - 1);
		calendar.set(Calendar.DATE, day);
		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, minute);
		calendar.set(Calendar.SECOND, seconds);
		calendar.set(Calendar.MILLISECOND, milliSecond);
		return calendar.getTime();
	}

	/**
	 * Creates the random date.
	 *
	 * @param from
	 *            the from
	 * @return the date
	 */
	public static Date newRandomDate(final Date from)
	{
		final Random secrand = new SecureRandom();
		final double randDouble = -secrand.nextDouble() * from.getTime();
		final double randomDouble = from.getTime() - secrand.nextDouble();
		final double result = randDouble * randomDouble;
		return new Date((long)result);
	}

	/**
	 * Creates a date from the current system time.
	 *
	 * @return the date
	 */
	public static Date now()
	{
		return new Date(System.currentTimeMillis());
	}

	/**
	 * Util-class. Do not make instances from this class.
	 */
	private CreateDateUtils()
	{
	}

}
