/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.date;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;


/**
 * Utility class for the use of Date and Calendar object.
 *
 * @version 1.0
 * @author Asterios Raptis
 */
public class DateUtils implements DatePatterns {

    /**
     * Returns a list with all dateformats from the Interface DateFormats.
     *
     * @return Returns a list with all dateformats from the Interface
     *         DateFormats.
     */
    public static List<String> getAllDateFormats() {
        final Field [] fields = DatePatterns.class.getFields();
        final List<String> list = new ArrayList<String>( fields.length );
        for ( int i = 0; i < fields.length; i++ ) {
            try {
                list.add( (String) fields[ i ].get( fields[ i ].getName() ) );
            } catch ( final IllegalArgumentException e ) {
                e.printStackTrace();
            } catch ( final IllegalAccessException e ) {
                e.printStackTrace();
            } finally {
            }
        }
        return list;
    }

    /**
     * Returns a map with all datepatterns from the Interface DatePatterns. As
     * key is the name from the pattern.
     *
     * @return Returns a Map with all datepatterns from the Interface
     *         DatePatterns.
     */
    public static Map< String, Object > getAllDatePatterns() {
        final Field [] fields = DatePatterns.class.getFields();
        final Map< String, Object > patterns = new HashMap< String, Object >(
                fields.length );
        for ( int i = 0; i < fields.length; i++ ) {
            try {
                patterns.put( fields[ i ].getName(), fields[ i ]
                        .get( fields[ i ].getName() ) );
            } catch ( final IllegalArgumentException e ) {
                e.printStackTrace();
            } catch ( final IllegalAccessException e ) {
                e.printStackTrace();
            } finally {
            }
        }
        return patterns;
    }

    /**
     * The Method from the Date.getDay is deprecated. This is a helper-Method.
     *
     * @param date
     *            The Date-object to get the day.
     * @return The day from the Date-object.
     */
    public static int getDay( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.DAY_OF_MONTH );
    }

    /**
     * The Method from the Date.getHours is deprecated. This is a helper-Method.
     *
     * @param date
     *            The Date-object to get the hours.
     * @return The hours from the Date-object.
     */
    public static int getHours( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.HOUR_OF_DAY );
    }

    /**
     * The Method from the Date.getMinutes is deprecated. This is a
     * helper-Method.
     *
     * @param date
     *            The Date-object to get the minutes.
     * @return The minutes from the Date-object.
     */
    public static int getMinutes( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.MINUTE );
    }

    /**
     * The Method from the Date.getMonth is deprecated. This is a helper-Method.
     *
     * @param date
     *            The Date-object to get the month.
     * @return The month from the Date-object.
     */
    public static int getMonth( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.MONTH );
    }

    /**
     * The Method from the Date.getSeconds is deprecated. This is a
     * helper-Method.
     *
     * @param date
     *            The Date-object to get the seconds.
     * @return The seconds from the Date-object.
     */
    public static int getSeconds( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.SECOND );
    }

    /**
     * The year as int from the Date-object.
     *
     * @param date
     *            The Date-object to get the year.
     * @return The year from the Date-object.
     */
    public static int getYear( final Date date ) {
        final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );
        return calendar.get( Calendar.YEAR );
    }

    /**
     * The week of the year as an int from the given Date-object.
     *
     * @param date
     *            The Date-object to get the week of the year.
     * @return The week of the year from the Date-object.
     */
    public static int getWeekOfYear(Date date) {		
		final Calendar calendar = Calendar.getInstance();
        calendar.setTime( date );  	
		return calendar.get(Calendar.WEEK_OF_YEAR);
	}
    
    /**
     * Returns a new {@link Date} object from the given Date object and sets the given parameters.
     *
     * @param dateToSet the date to set
     * @param hours the hours
     * @param minutes the minutes
     * @param seconds the seconds
     * @param milisec the milisec
     * @param zone the zone
     * @param locale the a locale
     * @return the date
     */
    public static Date setDate(Date dateToSet, int hours, int minutes, int seconds, int milisec, TimeZone zone,
            Locale locale) {
    	Calendar cal = Calendar.getInstance(zone, locale);
    	cal.setTime(dateToSet);
    	cal.set(Calendar.HOUR_OF_DAY, hours);
    	cal.set(Calendar.MINUTE, minutes);
    	cal.set(Calendar.SECOND, seconds);
    	cal.set(Calendar.MILLISECOND, milisec);
    	return cal.getTime();
    }

    /**
     * Util-class. Do not make instances from this class.
     */
    private DateUtils() {
    }

}