package de.audiophobe;

import de.audiophobe.parser.JavaClassParser;
import de.audiophobe.parser.JavaSourceParser;
import java.io.File;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;


///p=CalciferMojo
@Mojo(name = "parse", requiresDependencyResolution = ResolutionScope.TEST)
public class CalciferMojo extends AbstractMojo {

  public static final String INCLUDE = "TIP: IMPORT HERE ";
  public static final String INCLUDE_FRAGMENT = "TIP: IMPORT FRAGMENT HERE ";

  @Parameter(defaultValue = "${project}", readonly = true, required = true)
  private MavenProject project;

  @Parameter(defaultValue = "${session}", readonly = true, required = true)
  private MavenSession session;

  private ClassLoader extendedLoader = null;

  public static List<String> ENCODINGS = Arrays.asList(new String[]{"UTF-8","ISO-8859-1","windows-1250"});
  ///* Comment 1.
  ///* Comment 2.
  ///* Comment 3.

  ///*Important* Comment

  private List<File> allFiles = new ArrayList<>();

  /***
   * Das hier ist Teil eines JavaDoc-Komentars.
   * Dieser kann sich über *mehrere* Zeilen erstrecken
   *
   * .und diese Liste enthalten
   * * Test1
   * * Test2
   */


  public boolean isValidFile(File f) {
    return f.isFile() && f.exists() && f.length() > 0 && (f.getName().endsWith(".java") || f.getName().endsWith(".properties"));
  }

  public boolean isValidDirectory(File f) {
    return f.isDirectory() && !f.getName().startsWith(".git") && !f.getName().startsWith("target") && !f.getName().startsWith(".idea");

  }

  public void listFiles(File root) {
    ///Das hier ist eine wichtige Zeile
    List<File> workDir = Arrays.asList(root.listFiles());

    for (File f : workDir) {
      ///Das hier ist noch eine wichtige Zeile
      if (isValidFile(f)) {
        allFiles.add(f);
        //getLog().info("Found: " + f.getAbsolutePath());
      } else if (isValidDirectory(f)) {
        listFiles(f);
      }
    }
  }

  public void execute() throws MojoExecutionException {
    try {
      getLog().info("Running in Basedir " + project.getBasedir().getAbsolutePath());
      addClasspath();
      listFiles(project.getBasedir());
      createMarkup();
    } catch (Exception e) {
      e.printStackTrace();
      getLog().error("Error in execution", e);
      throw new MojoExecutionException("Error execution", e);
    }
  }


  public String extractPage(String s) {
    String regex = "p=(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }

  public String extractBlockMarker(String s) {
    String regex = "b=(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }

  public String extractCodeBlockMarker(String s) {
    String regex = "cb=(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }


  public String extractIncludeMarker(String s) {
    String regex = "i=(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      return matcher.group(1);
    }
    return null;
  }


  public DiagramLine extractDiagramLine(String s) {
    String regex = "(dl?)=(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      String directive = matcher.group(1);
      String[] data = matcher.group(2).split(";");
      if (directive.equals("d")) {
        DiagramLine diagramLine = new DiagramLine();
        diagramLine.diagramName = data[0];
        diagramLine.line = data[1];
        return diagramLine;
      } else {
        DiagramLine diagramLine = new DiagramLine();
        diagramLine.diagramName = null;
        diagramLine.line = data[0];
        return diagramLine;

      }
    }
    return null;
  }

  public EnumLine extractEnumLine(String s) {
    String regex = "enum=(.*);(.*)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(s);
    if (matcher.matches()) {
      EnumLine line = new EnumLine();
      line.enumName = matcher.group(1);
      line.fields = matcher.group(2);
      return line;
    }
    return null;
  }

  /**
   * Das ist ein Test
   */

  public void createMarkupForFile(File f, CalciferProject calciferProject) {
    boolean headerCreated = false;
    getLog().info("Create markup for " + f.getAbsolutePath());
    List<List<String>> allBlocks = new ArrayList<>();

    List<String> lines = CalciferUtil.INSTANCE.readAllLines(f,getLog());

    try {
      List<String> allComments = new ArrayList<>();

      JavaSourceParser javaSourceParser = new JavaSourceParser();
      List<CalciferCodeLine> allCodeLines = javaSourceParser.extractLines(f,getLog());

      CalciferCodeStateE state = CalciferCodeStateE.NONE;

      CalciferBlock currentBlock = new CalciferBlock();

      String currentPage = CalciferProject.TOP;
      String currentBlockMarker = null;
      String currentDiagramName = null;

      for (CalciferCodeLine cl : allCodeLines) {
        String c = cl.getParseRemoveSpace();
        if (c != null) {
          String page = extractPage(c);
          String block = extractBlockMarker(c);
          String codeblock = extractCodeBlockMarker(c);
          String includeMarker = extractIncludeMarker(c);
          DiagramLine diagramLine = extractDiagramLine(c);
          EnumLine enumLine = extractEnumLine(c);

          if (enumLine != null) {
            CalciferEnum calciferEnum = calciferProject.getEnum(enumLine.enumName);
            calciferEnum.setClassname(enumLine.enumName);
            calciferEnum.setFields(Arrays.asList(enumLine.fields.split(",")));
            calciferEnum.setClassLoader(extendedLoader);
          } else if (includeMarker != null) {
            currentBlock.add(INCLUDE + includeMarker);
          } else if (diagramLine != null) {
            if (diagramLine.diagramName != null) {
              currentDiagramName = diagramLine.diagramName;
            } else {
              diagramLine.diagramName = currentDiagramName;
            }
            CalciferDiagram calciferDiagram = calciferProject.getDiagram(currentDiagramName);
            calciferDiagram.add(diagramLine.line);
            getLog().info("Diagram: " + currentDiagramName + ", line: " + diagramLine.line);
          } else if (block != null) {
            state = CalciferCodeStateE.BLOCK;
            currentBlockMarker = block;
            currentBlock = calciferProject.getBlock(currentBlockMarker);
          } else if (codeblock != null) {
            state = CalciferCodeStateE.CODEBLOCK;
            currentBlockMarker = codeblock;
            currentBlock = calciferProject.getBlock(currentBlockMarker);
            currentBlock.add("....");
          } else if (page != null) {
            currentPage = page;
            //getLog().info("New Page: " + page);
          } else {
            if (state == CalciferCodeStateE.CODEBLOCK) {
              currentBlock.add(cl.getParseWithSpace());
            } else {
              currentBlock.add(c);
            }
          }
        } else {
          if (!currentBlock.isEmpty()) {
            if (state == CalciferCodeStateE.CODEBLOCK) {
              currentBlock.add("....");
            }

            if (currentBlockMarker != null) {
              calciferProject.blocks.put(currentBlockMarker, currentBlock);
            } else {
              calciferProject.addBlockToPage(currentPage, currentBlock);
            }

            currentBlock = new CalciferBlock();
            state = CalciferCodeStateE.NONE;
            currentBlockMarker = null;
          }
        }
      }

      if (!currentBlock.isEmpty()) {
        calciferProject.addBlockToPage(currentPage, currentBlock);
      }
    } catch (Exception e) {
      getLog().error(e);
      throw new RuntimeException(e);
    }
    getLog().info("Done with " + f.getAbsolutePath());
  }

  public void writePage(File dir, String filename, String content) {
    dir.mkdirs();
    File f = new File(dir, filename);
    try {
      Files.write(f.toPath(), content.getBytes(StandardCharsets.UTF_8));
    } catch (Exception e) {
      getLog().error(e);
      throw new RuntimeException(e);
    }
  }

  public String replaceFields(final String s, final CalciferProject calciferProject) {
    String result = s;
    List<String> placeholders = getPlaceholders(result);
    if (result != null && !placeholders.isEmpty()) {
      for (String ph : placeholders) {
        String toReplace = "${" + ph + "}";
        getLog().info("Check placeholder " + toReplace);
        CalciferPart calciferPart = calciferProject.findPart(ph);
        if (calciferPart != null) {
          String replaceWith = calciferPart.getContents(true, calciferProject).stream().collect(Collectors.joining("\n"));
          getLog().info("Replace " + toReplace + " with " + replaceWith);
          result = result.replace(toReplace, replaceWith);
        }
      }
    }
    return result;
  }

  public List<String> getPlaceholders(final String s) {
    List<String> result = new ArrayList<>();
    if (s != null && !s.isEmpty()) {
      String pattern = "\\$\\{([^\\}]+)\\}";
      Pattern p = Pattern.compile(pattern);
      Matcher m = p.matcher(s);
      while (m.find()) {
        String ph = m.group(1);
        getLog().info("Found placeholder " + ph);
        result.add(ph);
      }
    }
    return result;
  }

  public void generateMarkup(CalciferProject calciferProject) {
    for (String pageName : calciferProject.pages.keySet()) {
      getLog().info("Generate markup für pageName " + pageName);
      CalciferPage calciferPage = calciferProject.getPage(pageName);

      File dir = null;

      if (pageName.equals(CalciferProject.TOP)) {
        dir = calciferProject.rootDir;
      } else {
        dir = new File(calciferProject.rootDir, "top-level-page");
        dir.mkdirs();
      }

      String title = calciferPage.getPageName();
      String filename = calciferPage.getFilename();

      //getLog().info("Generate markup für pageName " + pageName + ", title: " + title + ", filename: " + filename);

      StringBuilder sb = new StringBuilder();

      sb.append("= " + title + "\n\n");

      List<String> allContent = calciferPage.getContents(false, calciferProject);

      while (true) {
        List<String> allContentCollected = new ArrayList<>();

        boolean include = false;

        for (String line : allContent) {
          if (line.startsWith(INCLUDE)) {
            String partId = line.replace(INCLUDE, "");
            getLog().info("Include " + partId);
            CalciferPart part = calciferProject.findPart(partId);
            List<String> partContent = part.getContents(false, calciferProject);
            allContentCollected.addAll(partContent);
            getLog().info("Added " + partContent.size() + " from " + partId);
            include = true;
          } else if (line.startsWith(INCLUDE_FRAGMENT)) {
            String partId = line.replace(INCLUDE_FRAGMENT, "");
            getLog().info("Include fragment " + partId);
            CalciferPart part = calciferProject.findPart(partId);
            List<String> partContent = part.getContents(true, calciferProject);
            allContentCollected.addAll(partContent);
            getLog().info("Added " + partContent.size() + " from " + partId);
            include = true;
          } else {
            String replacedFields = replaceFields(line, calciferProject);
            allContentCollected.add(replacedFields);
            if (!line.equals(replacedFields)) {
              include = true;
            }
          }
        }

        allContent = allContentCollected;

        if (include == false) {
          break;
        }
      }

      for (String line : allContent) {
        sb.append(line);
        sb.append("\n");
      }

      String content = sb.toString();
      writePage(dir, filename, content);
    }
  }

  boolean deleteDirectory(File directoryToBeDeleted) {
    File[] allContents = directoryToBeDeleted.listFiles();
    if (allContents != null) {
      for (File file : allContents) {
        deleteDirectory(file);
      }
    }
    return directoryToBeDeleted.delete();
  }

  public void createMarkup() {
    StringBuilder sb = new StringBuilder();
    File rootDir = new File(project.getBasedir(), "/confluence");

    if (rootDir.exists()) {
      try {
        deleteDirectory(rootDir);
      } catch (Exception e) {
        getLog().error(e);
        throw new RuntimeException(e);
      }
    }

    sb.append("= Generierte Doku zu " + project.getName() + "\n");

    CalciferProject calciferProject = new CalciferProject();
    calciferProject.rootDir = rootDir;
    calciferProject.title = project.getName();

    JavaClassParser classParser = new JavaClassParser();

    for (File f : allFiles) {
      classParser.parseClass(f, calciferProject,getLog(),extendedLoader);
    }

    for (File f : allFiles) {
      createMarkupForFile(f, calciferProject);
    }

    generateMarkup(calciferProject);
  }


  public void addClasspath() {
    try {
      Set<URL> urls = new HashSet<>();
      Set<String> elements = new HashSet<>();

      elements.addAll(project.getCompileClasspathElements());
      elements.addAll(project.getRuntimeClasspathElements());
      elements.addAll(project.getTestClasspathElements());

      getLog().info("Check deps");
      for (Dependency dependency : project.getDependencies()) {
        getLog().info(dependency.getSystemPath());
        getLog().info("A:" + dependency.getArtifactId());
      }

      for (String element : elements) {
        getLog().info("Add Classpath Element: " + element);
        urls.add(new File(element).toURI().toURL());
      }

      extendedLoader = URLClassLoader.newInstance(
          urls.toArray(new URL[0]),
          Thread.currentThread().getContextClassLoader());
    } catch (Exception e) {
      getLog().error(e);
      throw new RuntimeException(e);
    }
  }


}
