package de.charlex.compose.bottomdrawerscaffold

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxWithConstraints
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ColumnScope
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.exclude
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.statusBars
import androidx.compose.foundation.layout.systemBars
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.BottomSheetScaffold
import androidx.compose.material3.BottomSheetScaffoldState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FabPosition
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SheetState
import androidx.compose.material3.SheetValue
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Surface
import androidx.compose.material3.contentColorFor
import androidx.compose.material3.rememberBottomSheetScaffoldState
import androidx.compose.material3.rememberStandardBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.coroutines.launch


@Composable
@ExperimentalMaterial3Api
fun rememberBottomSheetScaffoldState(
    bottomSheetState: SheetState = rememberStandardBottomSheetState(),
    snackbarHostState: SnackbarHostState = remember { SnackbarHostState() },
    onSheetStateChanged: (SheetValue) -> Unit
): BottomSheetScaffoldState {
    val state = remember(bottomSheetState, snackbarHostState) {
        BottomSheetScaffoldState(
            bottomSheetState = bottomSheetState,
            snackbarHostState = snackbarHostState
        )
    }

    LaunchedEffect(state.bottomSheetState.currentValue) {
        onSheetStateChanged(state.bottomSheetState.currentValue)
    }

    return state
}

@Composable
@ExperimentalMaterial3Api
fun BottomDrawerScaffold(
    modifier: Modifier = Modifier,
    contentWindowInsets: WindowInsets = WindowInsets.systemBars.exclude(WindowInsets.statusBars).exclude(WindowInsets.navigationBars),
    bottomSheetScaffoldState: BottomSheetScaffoldState = rememberBottomSheetScaffoldState(),
    topBar: (@Composable () -> Unit)? = null,
    bottomBar: @Composable (() -> Unit)? = null,
    gesturesEnabled: Boolean = true,
    drawerModifier: Modifier = Modifier,
    snackbarHost: @Composable (SnackbarHostState) -> Unit = {},
    floatingActionButton: @Composable (() -> Unit)? = null,
    floatingActionButtonPosition: FabPosition = FabPosition.End,
    drawerGesturesEnabled: Boolean? = null,
    drawerShape: Shape = RoundedCornerShape(topStart = 20.dp, topEnd = 20.dp),
    drawerPadding: Dp = 10.dp,
    drawerScrimColor: Color = BottomDrawerScaffoldDefaults.scrimColor(),
    drawerTonalElevation: Dp = BottomDrawerScaffoldDefaults.DrawerTonalElevation,
    drawerShadowElevation: Dp = BottomDrawerScaffoldDefaults.DrawerShadowElevation,
    drawerBackgroundColor: Color = MaterialTheme.colorScheme.surface,
    drawerContentColor: Color = MaterialTheme.colorScheme.contentColorFor(drawerBackgroundColor),
    drawerPeekHeight: Dp = BottomDrawerScaffoldDefaults.DrawerPeekHeight,
    drawerContent: @Composable ColumnScope.() -> Unit,
    backgroundColor: Color = MaterialTheme.colorScheme.background,
    contentColor: Color = MaterialTheme.colorScheme.contentColorFor(backgroundColor),
    content: @Composable (PaddingValues) -> Unit
) {
    Scaffold(
        modifier = modifier,
        contentWindowInsets = contentWindowInsets,
        topBar = {
            topBar?.invoke()
        },
        bottomBar = {
            bottomBar?.invoke()
        },
        floatingActionButton = {
            floatingActionButton?.invoke()
        },
        floatingActionButtonPosition = floatingActionButtonPosition,
    ) { scaffoldPaddingValues ->
        val scope = rememberCoroutineScope()

        BoxWithConstraints(
            modifier = Modifier
                .padding(scaffoldPaddingValues)
        ) {
            val fullHeight = constraints.maxHeight.toFloat()
            var bottomDrawerHeight by remember { mutableStateOf(fullHeight) }
            var initialOffset by remember { mutableStateOf(-1f) }

            Surface(
                color = backgroundColor,
                contentColor = contentColor
            ) {
                Box(Modifier.fillMaxSize()) {
                    content(PaddingValues(bottom = drawerPeekHeight))

                    Scrim(
                        open = bottomSheetScaffoldState.isExpanded(),
                        onClose = {
                            if (gesturesEnabled) {
                                scope.launch { bottomSheetScaffoldState.collapse() }
                            }
                        },
                        fraction = {
                            try {
                                calculateFraction(initialOffset, 0f, bottomSheetScaffoldState.bottomSheetState.requireOffset())
                            } catch (exception: IllegalStateException) {
                                0f
                            }
                        },
                        color = drawerScrimColor
                    )
                }
            }

            Box(
                modifier = if(topBar == null) Modifier.windowInsetsPadding(WindowInsets.statusBars) else Modifier
            ) {
                BottomSheetScaffold(
                    modifier = Modifier,
                    scaffoldState = bottomSheetScaffoldState,
                    sheetDragHandle = null,
                    sheetSwipeEnabled = drawerGesturesEnabled ?: gesturesEnabled,
                    sheetContainerColor = Color.Transparent,
                    sheetPeekHeight = drawerPeekHeight,
                    sheetTonalElevation = 0.dp,
                    sheetShadowElevation = 0.dp,
                    snackbarHost = snackbarHost,
                    sheetContent = {
                        Surface(
                            Modifier
                                .fillMaxWidth()
                                .padding(
                                    top = drawerPadding,
                                    start = drawerPadding,
                                    end = drawerPadding
                                )
                                .onGloballyPositioned {
                                    if(initialOffset == -1f) {
                                        initialOffset = bottomSheetScaffoldState.bottomSheetState.requireOffset()
                                    }
                                    bottomDrawerHeight = it.size.height.toFloat()
                                }
                                .then(
                                    drawerModifier
                                ),
                            shape = drawerShape,
                            shadowElevation = drawerShadowElevation,
                            tonalElevation = drawerTonalElevation,
                            color = drawerBackgroundColor,
                            contentColor = drawerContentColor,
                            content = {
                                Column(
                                    content = {
                                        drawerContent()
                                    }
                                )
                            }
                        )
                    },
                    content = {}
                )
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
fun BottomSheetScaffoldState.isCollapsed(): Boolean {
    return bottomSheetState.hasPartiallyExpandedState
}

@OptIn(ExperimentalMaterial3Api::class)
fun BottomSheetScaffoldState.isExpanded(): Boolean {
    return bottomSheetState.hasExpandedState
}

@OptIn(ExperimentalMaterial3Api::class)
suspend fun BottomSheetScaffoldState.toggle() {
    if (bottomSheetState.targetValue == SheetValue.Expanded) {
        bottomSheetState.partialExpand()
    } else if (bottomSheetState.targetValue == SheetValue.PartiallyExpanded) {
        bottomSheetState.expand()
    }
}

@OptIn(ExperimentalMaterial3Api::class)
suspend fun BottomSheetScaffoldState.collapse() {
    bottomSheetState.partialExpand()
}

@OptIn(ExperimentalMaterial3Api::class)
suspend fun BottomSheetScaffoldState.expand() {
    bottomSheetState.expand()
}

internal fun calculateFraction(a: Float, b: Float, pos: Float): Float {
    if (a == b) return 0f
    return ((pos - a) / (b - a)).coerceIn(0f, 1f)
}
