package de.codecamp.messages;

import java.util.Map;


/**
 * A common interface that can be implemented by objects that should provide access to messages.
 */
public interface HasMessages
{

  /**
   * Returns the message for the given message key.
   *
   * @param messageKey
   *          the message key code
   * @return the resolved message or null
   * @see MessageKeyUtils
   */
  default String getMessage(String messageKey)
  {
    return getMessage(messageKey, new Object[0]);
  }

  /**
   * Returns the message for the given message key and arguments.
   *
   * @param messageKey
   *          the message key code
   * @param args
   *          either a regular array of message arguments; or a single element array containing a
   *          map of named arguments ({@code Map<String, Object>}) or a {@link ResolvableMessage};
   *          may be null
   * @return the resolved message or null
   * @see MessageKeyUtils
   */
  String getMessage(String messageKey, Object... args);

  /**
   * Returns the message for the given message key and arguments.
   *
   * @param messageKey
   *          the message key code
   * @param args
   *          the map of named arguments; may be null
   * @return the resolved message or null
   * @see MessageKeyUtils
   */
  default String getMessage(String messageKey, Map<String, Object> args)
  {
    return getMessage(messageKey, args != null ? new Object[] {args} : new Object[0]);
  }

  /**
   * Returns the message for the given message key.
   *
   * @param messageKey
   *          the message key code
   * @return the resolved message or null
   */
  default String getMessage(MessageKey messageKey)
  {
    return getMessage(messageKey.getCode());
  }

  /**
   * Returns the message for the given {@link ResolvableMessage}.
   *
   * @param resolvableMessage
   *          the resolvable message
   * @return the resolved message or null
   */
  default String getMessage(ResolvableMessage resolvableMessage)
  {
    return getMessage(resolvableMessage.getCode(), resolvableMessage);
  }

}
