package de.codecamp.messages.runtime;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.function.Supplier;

import com.ibm.icu.util.Calendar;
import com.ibm.icu.util.TimeZone;


/**
 * Converts several java.time types and {@link java.util.Calendar} to {@link Calendar}.
 * <p>
 * If changes are made to the mapped types, this must be reflected in
 * {@code de.codecamp.messages.shared.validation.IcuMessageFormatCheck}.
 */
public class IcuMessageArgConverter
  implements
    MessageArgConverter
{

  private final Supplier<java.util.TimeZone> timeZoneProvider;


  public IcuMessageArgConverter(Supplier<java.util.TimeZone> timeZoneProvider)
  {
    this.timeZoneProvider = timeZoneProvider;
  }


  @Override
  public Object convert(Object value)
  {
    ZonedDateTime zonedDateTime = null;
    if (value instanceof java.util.Calendar)
    {
      java.util.Calendar juc = (java.util.Calendar) value;
      zonedDateTime = ZonedDateTime.ofInstant(juc.toInstant(), juc.getTimeZone().toZoneId());
    }
    else if (value instanceof LocalDate)
    {
      zonedDateTime = ((LocalDate) value).atStartOfDay(zoneId());
    }
    else if (value instanceof LocalTime)
    {
      zonedDateTime = ((LocalTime) value).atDate(LocalDate.now(zoneId())).atZone(zoneId());
    }
    else if (value instanceof LocalDateTime)
    {
      zonedDateTime = ((LocalDateTime) value).atZone(zoneId());
    }
    else if (value instanceof ZonedDateTime)
    {
      zonedDateTime = (ZonedDateTime) value;
    }
    else if (value instanceof OffsetTime)
    {
      zonedDateTime =
          ((OffsetTime) value).atDate(LocalDate.now(zoneId())).atZoneSameInstant(zoneId());
    }
    else if (value instanceof OffsetDateTime)
    {
      zonedDateTime = ((OffsetDateTime) value).atZoneSameInstant(zoneId());
    }
    else if (value instanceof Instant)
    {
      zonedDateTime = ((Instant) value).atZone(zoneId());
    }

    if (zonedDateTime != null)
    {
      Calendar calendar =
          Calendar.getInstance(TimeZone.getTimeZone(timeZoneProvider.get().getID()));
      calendar.setTimeInMillis(zonedDateTime.toInstant().toEpochMilli());
      return calendar;
    }

    return value;
  }

  private ZoneId zoneId()
  {
    return timeZoneProvider.get().toZoneId();
  }

}
