package de.codecamp.messages;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;

import de.codecamp.messages.runtime.MessageArgConverter;


/**
 * Contains a message key code and the message arguments as named and indexed values to resolve it.
 * But even named arguments have an implicit order (and therefore index) based on the order they are
 * declared in for the message key. Use {@link #forCode(String)} to create new instances.
 */
public class ResolvableMessage
  implements
    MessageKey
{

  private final String code;

  private final Map<String, Object> argsByName;

  private final List<Object> argsByIndex;


  private ResolvableMessage(String code)
  {
    this.code = code;
    this.argsByName = Collections.emptyMap();
    this.argsByIndex = Collections.emptyList();
  }

  private ResolvableMessage(String code, Map<String, Object> argsByName, List<Object> argsByIndex)
  {
    this.code = code;
    this.argsByName = argsByName;
    this.argsByIndex = argsByIndex;
  }


  @Override
  public String getCode()
  {
    return code;
  }

  public Map<String, Object> getArgsByName()
  {
    return getArgsByName(false);
  }

  public Map<String, Object> getArgsByName(boolean includeByIndex)
  {
    if (!includeByIndex)
    {
      if (argsByName.size() != argsByIndex.size())
        throw new IllegalStateException("Names aren't available for all message arguments.");

      return argsByName;
    }

    Map<String, Object> result = new HashMap<>(argsByName);
    for (int i = 0; i < argsByIndex.size(); i++)
      result.put(Integer.toString(i), argsByIndex.get(i));
    return result;
  }

  public Object[] getArgsByIndex()
  {
    return argsByIndex.toArray(new Object[argsByIndex.size()]);
  }


  public void convertValues(MessageArgConverter converter, Locale locale)
  {
    for (Map.Entry<String, Object> entry : argsByName.entrySet())
    {
      entry.setValue(converter.convert(entry.getValue(), locale));
    }
    for (ListIterator<Object> it = argsByIndex.listIterator(); it.hasNext();)
    {
      it.set(converter.convert(it.next(), locale));
    }
  }


  public static Builder forCode(String code)
  {
    return new Builder(code);
  }

  public static ResolvableMessage forCodeNoArgs(String code)
  {
    return new ResolvableMessage(code);
  }


  public static class Builder
  {

    private final String code;

    private Map<String, Object> argsByName;

    private List<Object> argsByIndex;


    public Builder(String code)
    {
      this.code = code;
    }


    public Builder arg(String name, Object value)
    {
      if (argsByName == null)
      {
        argsByName = new HashMap<>();
        argsByIndex = new ArrayList<>();
      }

      if (name != null)
        argsByName.put(name, value);
      argsByIndex.add(value);
      return this;
    }


    public ResolvableMessage build()
    {
      return new ResolvableMessage(code,
          argsByName == null || argsByName.isEmpty() ? Collections.emptyMap()
              : new HashMap<>(argsByName),
          argsByIndex == null || argsByIndex.isEmpty() ? Collections.emptyList()
              : new ArrayList<>(argsByIndex));
    }

  }

}
