package de.codecamp.messages;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Default implementation of {@link ResolvableMessage}.
 */
class ResolvableMessageImpl
  implements
    ResolvableMessage
{

  private final String code;

  private final Map<String, Object> argsByName;

  private final List<Object> argsByIndex;


  ResolvableMessageImpl(String code)
  {
    this.code = code;
    this.argsByName = Collections.emptyMap();
    this.argsByIndex = Collections.emptyList();
  }

  ResolvableMessageImpl(String code, Map<String, Object> argsByName, List<Object> argsByIndex)
  {
    this.code = code;
    this.argsByName = Collections.unmodifiableMap(argsByName);
    this.argsByIndex = Collections.unmodifiableList(argsByIndex);
  }


  @Override
  public String getCode()
  {
    return code;
  }

  @Override
  public Map<String, Object> getArgsByName()
  {
    return getArgsByName(false);
  }

  @Override
  public Map<String, Object> getArgsByName(boolean includeByIndex)
  {
    if (!includeByIndex)
    {
      if (argsByName.size() != argsByIndex.size())
        throw new IllegalStateException("Names aren't available for all message arguments.");

      return argsByName;
    }

    Map<String, Object> result = new HashMap<>(argsByName);
    for (int i = 0; i < argsByIndex.size(); i++)
      result.put(Integer.toString(i), argsByIndex.get(i));
    return result;
  }

  @Override
  public List<Object> getArgsByIndex()
  {
    return argsByIndex;
  }


  public static Builder forCode(String code)
  {
    return new Builder(code);
  }

  public static ResolvableMessageImpl forCodeNoArgs(String code)
  {
    return new ResolvableMessageImpl(code);
  }


  public static class Builder
  {

    private final String code;

    private Map<String, Object> argsByName;

    private List<Object> argsByIndex;


    public Builder(String code)
    {
      this.code = code;
    }


    public Builder arg(String name, Object value)
    {
      if (argsByName == null)
      {
        argsByName = new HashMap<>();
        argsByIndex = new ArrayList<>();
      }

      if (name != null)
        argsByName.put(name, value);
      argsByIndex.add(value);
      return this;
    }


    public ResolvableMessageImpl build()
    {
      return new ResolvableMessageImpl(code,
          argsByName == null || argsByName.isEmpty() ? Collections.emptyMap()
              : new HashMap<>(argsByName),
          argsByIndex == null || argsByIndex.isEmpty() ? Collections.emptyList()
              : new ArrayList<>(argsByIndex));
    }

  }

}
