package de.codecamp;

import static java.util.stream.Collectors.toList;
import static javax.lang.model.util.ElementFilter.fieldsIn;
import static javax.lang.model.util.ElementFilter.methodsIn;

import java.beans.Introspector;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.annotation.Repeatable;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Stream;

import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Elements;
import javax.tools.FileObject;
import javax.tools.JavaFileManager.Location;
import javax.tools.StandardLocation;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;


public class ProcessorUtils
{

  private static final String GET_PREFIX = "get";

  private static final String IS_PREFIX = "is";

  private static final Set<String> IGNORED_PROPERTIES = new HashSet<>(Arrays.asList("class"));


  private ProcessingEnvironment processingEnv;


  public ProcessorUtils(ProcessingEnvironment processingEnv)
  {
    this.processingEnv = processingEnv;
  }


  public String getOption(String name)
  {
    return processingEnv.getOptions().get(name);
  }

  public String getOption(String name, String defaultValue)
  {
    String value = getOption(name);
    if (StringUtils.isEmpty(value))
      value = defaultValue;
    return value;
  }

  public Boolean getOptionAsBoolean(String name)
  {
    return Boolean.valueOf(getOption(name));
  }

  public Boolean getOptionAsBoolean(String name, Boolean defaultValue)
  {
    Boolean value = getOptionAsBoolean(name);
    if (value == null)
      value = defaultValue;
    return value;
  }

  public List<String> getOptionAsList(String name)
  {
    String rawValue = processingEnv.getOptions().get(name);
    if (rawValue == null)
      return null;
    return Stream.of(StringUtils.splitPreserveAllTokens(rawValue, ",")).map(String::trim)
        .collect(toList());
  }

  public List<String> getOptionAsList(String name, List<String> defaultValues)
  {
    List<String> value = getOptionAsList(name);
    if (value == null || value.isEmpty())
      value = defaultValues;
    return value;
  }

  public <T extends Enum<T>> T getOptionAsEnum(String name, Class<T> enumType)
  {
    String value = getOption(name);
    if (StringUtils.isEmpty(value))
      return null;
    else
      return Enum.valueOf(enumType, value);
  }

  public <T extends Enum<T>> T getOptionAsEnum(String name, Class<T> enumType, T defaultValue)
  {
    T e = getOptionAsEnum(name, enumType);
    if (e == null)
      e = defaultValue;
    return e;
  }


  public static TypeElement getGeneratedAnnotation(Elements elements, SourceVersion sourceVersion)
  {
    TypeElement typeElement = null;

    typeElement = elements.getTypeElement("javax.annotation.processing.Generated");
    if (typeElement == null)
      typeElement = elements.getTypeElement("javax.annotation.Generated");

    /*
     * ProcessingEnvironment#getSourceVersion() does not seem to be reliable when used in Eclipse.
     * Was always RELEASE_7 no matter what was configured for the project.
     */
    // if (sourceVersion.compareTo(SourceVersion.RELEASE_8) > 0)
    // typeElement = elements.getTypeElement("javax.annotation.processing.Generated");
    // else
    // typeElement = elements.getTypeElement("javax.annotation.Generated");

    return typeElement;
  }

  public static PackageElement getPackage(Element element)
  {
    while (element != null && element.getKind() != ElementKind.PACKAGE)
    {
      element = element.getEnclosingElement();
    }
    return (PackageElement) element;
  }

  public static List<ExecutableElement> getDeclaredMethods(TypeElement element)
  {
    return methodsIn(element.getEnclosedElements());
  }

  public static List<VariableElement> getDeclaredFields(TypeElement element)
  {
    return fieldsIn(element.getEnclosedElements());
  }

  public static List<VariableElement> getEnumConstants(TypeElement enumTypeElement)
  {
    if (enumTypeElement.getKind() != ElementKind.ENUM)
      throw new IllegalArgumentException("not an enum: " + enumTypeElement.getQualifiedName());

    return fieldsIn(enumTypeElement.getEnclosedElements()).stream()
        .filter(ve -> ve.getKind() == ElementKind.ENUM_CONSTANT).collect(toList());
  }


  public static List<TypeElement> getTypeNesting(TypeElement typeElement)
  {
    List<TypeElement> result = new ArrayList<>();

    while (typeElement != null)
    {
      result.add(0, typeElement);

      typeElement = getEnclosingTypeElement(typeElement);
    }
    return result;
  }

  public static TypeElement getEnclosingTypeElement(Element element)
  {
    do
    {
      element = element.getEnclosingElement();
    }
    while (element != null && !element.getKind().isClass() && !element.getKind().isInterface());

    return (TypeElement) element;
  }


  /**
   * If the annotation is repeatable, this method will also find elements with the container
   * annotation.
   *
   * @see RoundEnvironment#getElementsAnnotatedWith(Class)
   */
  public Set<? extends Element> getElementsAnnotatedWith(RoundEnvironment roundEnv,
      Class<? extends Annotation> annotationType)
  {
    Set<Element> result = new HashSet<>();

    result.addAll(roundEnv.getElementsAnnotatedWith(annotationType));

    Repeatable repeatableAt = annotationType.getAnnotation(Repeatable.class);
    if (repeatableAt != null)
    {
      result.addAll(roundEnv.getElementsAnnotatedWith(repeatableAt.value()));
    }

    return result;
  }

  public static AnnotationMirror getAnnotationMirror(Element element,
      Class<? extends Annotation> annotationType)
  {
    for (AnnotationMirror am : element.getAnnotationMirrors())
    {
      if (am.getAnnotationType().toString().equals(annotationType.getName()))
      {
        return am;
      }
    }
    return null;
  }

  public Set<AnnotationMirror> getAnnotationMirrors(Element element,
      Class<? extends Annotation> annotationType)
  {
    Class<? extends Annotation> containerAnnotationType = null;
    Repeatable repeatableAt = annotationType.getAnnotation(Repeatable.class);
    if (repeatableAt != null)
    {
      containerAnnotationType = repeatableAt.value();
    }

    Set<AnnotationMirror> result = new HashSet<>();
    for (AnnotationMirror am : element.getAnnotationMirrors())
    {
      if (containerAnnotationType != null
          && am.getAnnotationType().toString().equals(containerAnnotationType.getCanonicalName()))
      {
        result.addAll(getAnnotationValueAsAnnotationMirrors(am, "value"));
      }
      else if (am.getAnnotationType().toString().equals(annotationType.getName()))
      {
        result.add(am);
      }
    }
    return result;
  }


  public AnnotationValue getAnnotationValue(AnnotationMirror annotationMirror, String key)
  {
    for (Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : annotationMirror
        .getElementValues().entrySet())
    {
      if (entry.getKey().getSimpleName().toString().equals(key))
      {
        return entry.getValue();
      }
    }
    return null;
  }

  public <T> T getAnnotationValueAs(AnnotationMirror annotationMirror, String key, Class<T> type,
      T defaultValue)
  {
    AnnotationValue annotationValue = getAnnotationValue(annotationMirror, key);

    Object value;
    if (annotationValue != null)
      value = annotationValue.getValue();
    else
      value = defaultValue;

    return type.cast(value);
  }

  public <T> List<T> getAnnotationValuesAs(AnnotationMirror annotationMirror, String key,
      Class<T> listElementType)
  {
    List<AnnotationValue> annotationValues = getAnnotationValues(annotationMirror, key);

    if (annotationValues == null)
      return null;

    return annotationValues.stream().map(av -> listElementType.cast(av.getValue()))
        .collect(toList());
  }

  @SuppressWarnings("unchecked")
  public List<AnnotationValue> getAnnotationValues(AnnotationMirror annotationMirror, String key)
  {
    for (Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : annotationMirror
        .getElementValues().entrySet())
    {
      if (entry.getKey().getSimpleName().toString().equals(key))
      {
        AnnotationValue annotationValue = getAnnotationValue(annotationMirror, key);
        return (List<AnnotationValue>) annotationValue.getValue();
      }
    }
    // empty arrays in annotations are not contained among element values
    return Collections.emptyList();
  }


  public TypeElement getAnnotationValueAsType(AnnotationMirror annotationMirror, String key)
  {
    AnnotationValue annotationValue = getAnnotationValue(annotationMirror, key);
    if (annotationValue == null)
      return null;

    TypeMirror typeMirror = (TypeMirror) annotationValue.getValue();
    if (typeMirror == null)
      return null;

    return (TypeElement) processingEnv.getTypeUtils().asElement(typeMirror);
  }

  public List<TypeElement> getAnnotationValueAsTypes(AnnotationMirror annotationMirror, String key)
  {
    List<AnnotationValue> values = getAnnotationValues(annotationMirror, key);
    if (values == null)
      return null;

    return values.stream()
        .map(av -> (TypeElement) processingEnv.getTypeUtils().asElement((TypeMirror) av.getValue()))
        .collect(toList());
  }

  public List<AnnotationMirror> getAnnotationValueAsAnnotationMirrors(
      AnnotationMirror annotationMirror, String key)
  {
    List<AnnotationValue> values = getAnnotationValues(annotationMirror, key);
    if (values == null)
      return null;

    return values.stream().map(av -> (AnnotationMirror) av.getValue()).collect(toList());
  }


  public static <A extends TypeElement> Set<A> findComposedAnnotationTypes(
      Set<A> processedAnnotations, Class<? extends Annotation> metaAnnotationType,
      boolean includeMetaAnnotation)
  {
    Set<A> annotations = new HashSet<>();
    for (A annotationElement : processedAnnotations)
    {
      if ((includeMetaAnnotation
          && annotationElement.getQualifiedName().toString().equals(metaAnnotationType.getName()))
          || annotationElement.getAnnotation(metaAnnotationType) != null)
      {
        annotations.add(annotationElement);
      }
    }
    return annotations;
  }

  public static <A extends Annotation> A findMetaAnnotation(Element element,
      Class<A> annotationType)
  {
    List<A> metaAnnotations = findMetaAnnotations(element, annotationType);
    if (!metaAnnotations.isEmpty())
      return metaAnnotations.get(0);
    else
      return null;
  }

  public static <A extends Annotation> List<A> findMetaAnnotations(Element element,
      Class<A> annotationType)
  {
    List<A> result = new ArrayList<>();

    A annotation = element.getAnnotation(annotationType);
    if (annotation != null)
      result.add(annotation);

    for (AnnotationMirror am : element.getAnnotationMirrors())
    {
      annotation = am.getAnnotationType().asElement().getAnnotation(annotationType);

      if (annotation != null)
        result.add(annotation);
    }

    return result;
  }

  public static <A extends Annotation> Map<AnnotationMirror, A> findMetaAnnotationsWithSource(
      Element element, Class<A> annotationType)
  {
    Map<AnnotationMirror, A> result = new LinkedHashMap<>();

    AnnotationMirror mirror = getAnnotationMirror(element, annotationType);
    A annotation = element.getAnnotation(annotationType);
    if (mirror != null && annotation != null)
      result.put(mirror, annotation);

    for (AnnotationMirror am : element.getAnnotationMirrors())
    {
      Element directAnnotationElement = am.getAnnotationType().asElement();

      mirror = getAnnotationMirror(directAnnotationElement, annotationType);
      annotation = directAnnotationElement.getAnnotation(annotationType);

      if (mirror != null && annotation != null)
        result.put(mirror, annotation);
    }

    return result;
  }


  public Path getSourceOutputPath()
  {
    return getOutputPath(StandardLocation.SOURCE_OUTPUT);
  }

  public Path getClassOutputPath()
  {
    return getOutputPath(StandardLocation.CLASS_OUTPUT);
  }

  private Path getOutputPath(Location location)
  {
    FileObject resource;
    try
    {
      resource =
          processingEnv.getFiler().createResource(location, "", UUID.randomUUID().toString());
      resource.delete();
      return Paths.get(resource.toUri()).getParent();
    }
    catch (IOException ex)
    {
      throw new RuntimeException(ex);
    }
  }


  public List<Triple<String, String, ExecutableElement>> findBeanProperties(
      TypeElement beanTypeElement)
  {
    List<ExecutableElement> methodElements = getDeclaredMethods(beanTypeElement);

    List<Triple<String, String, ExecutableElement>> properties = new ArrayList<>();

    for (ExecutableElement methodElement : methodElements)
    {
      if (!methodElement.getModifiers().contains(Modifier.PUBLIC))
        continue;
      if (methodElement.getModifiers().contains(Modifier.STATIC))
        continue;
      if (methodElement.getReturnType().getKind() == TypeKind.VOID)
        continue;
      if (!methodElement.getParameters().isEmpty())
        continue;

      String methodName = methodElement.getSimpleName().toString();

      if (methodName.startsWith(GET_PREFIX) || methodName.startsWith(IS_PREFIX))
      {
        String propertyName;
        if (methodName.startsWith(GET_PREFIX))
          propertyName = methodName.substring(GET_PREFIX.length());
        else
          propertyName = methodName.substring(IS_PREFIX.length());

        propertyName = Introspector.decapitalize(propertyName);

        if (IGNORED_PROPERTIES.contains(propertyName))
          continue;

        properties.add(Triple.of(propertyName, null, methodElement));
      }
    }

    return properties;
  }

}
