package de.codecamp.tracer;

import java.util.UUID;


/**
 * The {@link TraceContext} represents a certain area of code, like a method. Contexts can be
 * nested. Each context has a root {@link Trace} that can be further divided into sub-traces.
 */
public interface TraceContext
  extends
    AutoCloseable
{

  UUID getId();

  /**
   * Trace contexts can be nested. This method returns whether this is the root context.
   *
   * @return whether this is the root context
   */
  boolean isRootContext();

  /**
   * Setting a throwable indicates that the trace context hasn't exited regularly. As a consequence,
   * durations can't necessarily be compared to other traces.
   *
   * @param exitThrowable
   *          the throwable that exited the trace
   */
  void setExitThrowable(Throwable exitThrowable);


  /**
   * Starts a new sub-trace as a direct child of this trace context. Ends any currently active trace
   * of this context, if there is one.
   * <p>
   * This method returns an {@link ActiveTrace} that can be used in a try-with-resource.
   *
   * @param label
   *          the label to use for this trace
   * @return an {@link ActiveTrace} that can be used in a try-with-resource
   */
  default ActiveTrace startTrace(String label)
  {
    return startTrace(label, (Object[]) null);
  }

  /**
   * Starts a new sub-trace within this trace context. Ends the currently active trace of this trace
   * context, if there is one.
   * <p>
   * v *
   *
   * @param label
   *          the label to use for this trace; may contain placeholders
   * @param labelFormatArgs
   *          the format arguments for the placeholders in the label
   * @return an {@link ActiveTrace} that can be used in a try-with-resource
   */
  ActiveTrace startTrace(String label, Object... labelFormatArgs);

  /**
   * Returns the deepest currently active trace of the current context or a no-op dummy if none is
   * available.
   *
   * @return the deepest currently active trace of the current context or a no-op dummy if none is
   *         available.
   */
  ActiveTrace getActiveTrace();

  /**
   * Closes this trace context. Closing a trace context is the responsibility of the one opening it.
   * <p>
   * Overridden from {@link AutoCloseable#close()} to remove thrown {@link Exception}.
   */
  @Override
  void close();

}
