package physx.common;

import physx.NativeObject;
import physx.PlatformChecks;

/**
 * Descriptor used to create a PxCudaContextManager
 */
public class PxCudaContextManagerDesc extends NativeObject {

    static {
        PlatformChecks.requirePlatform(3, "physx.common.PxCudaContextManagerDesc");
    }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCudaContextManagerDesc wrapPointer(long address) {
        return address != 0L ? new PxCudaContextManagerDesc(address) : null;
    }
    
    public static PxCudaContextManagerDesc arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCudaContextManagerDesc(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxCudaContextManagerDesc
     */
    public static PxCudaContextManagerDesc createAt(long address) {
        __placement_new_PxCudaContextManagerDesc(address);
        PxCudaContextManagerDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxCudaContextManagerDesc
     */
    public static <T> PxCudaContextManagerDesc createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxCudaContextManagerDesc(address);
        PxCudaContextManagerDesc createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxCudaContextManagerDesc(long address);

    // Constructors

    public PxCudaContextManagerDesc() {
        address = _PxCudaContextManagerDesc();
    }
    private static native long _PxCudaContextManagerDesc();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * * The CUDA context to manage
     * *
     * * If left NULL, the PxCudaContextManager will create a new context.  If
     * * graphicsDevice is also not NULL, this new CUDA context will be bound to
     * * that graphics device, enabling the use of CUDA/Graphics interop features.
     * *
     * * If ctx is not NULL, the specified context must be applied to the thread
     * * that is allocating the PxCudaContextManager at creation time (aka, it
     * * cannot be popped).  The PxCudaContextManager will take ownership of the
     * * context until the manager is released.  All access to the context must be
     * * gated by lock acquisition.
     * *
     * * If the user provides a context for the PxCudaContextManager, the context
     * * _must_ have either been created on the GPU ordinal returned by
     * * PxGetSuggestedCudaDeviceOrdinal() or on your graphics device.
     */
    public CUcontext getCtx() {
        checkNotNull();
        return CUcontext.wrapPointer(_getCtx(address));
    }
    private static native long _getCtx(long address);

    /**
     * * The CUDA context to manage
     * *
     * * If left NULL, the PxCudaContextManager will create a new context.  If
     * * graphicsDevice is also not NULL, this new CUDA context will be bound to
     * * that graphics device, enabling the use of CUDA/Graphics interop features.
     * *
     * * If ctx is not NULL, the specified context must be applied to the thread
     * * that is allocating the PxCudaContextManager at creation time (aka, it
     * * cannot be popped).  The PxCudaContextManager will take ownership of the
     * * context until the manager is released.  All access to the context must be
     * * gated by lock acquisition.
     * *
     * * If the user provides a context for the PxCudaContextManager, the context
     * * _must_ have either been created on the GPU ordinal returned by
     * * PxGetSuggestedCudaDeviceOrdinal() or on your graphics device.
     */
    public void setCtx(CUcontext value) {
        checkNotNull();
        _setCtx(address, value.getAddress());
    }
    private static native void _setCtx(long address, long value);

    /**
     * * D3D device pointer or OpenGl context handle
     * *
     * * Only applicable when ctx is NULL, thus forcing a new context to be
     * * created.  In that case, the created context will be bound to this
     * * graphics device.
     */
    public NativeObject getGraphicsDevice() {
        checkNotNull();
        return NativeObject.wrapPointer(_getGraphicsDevice(address));
    }
    private static native long _getGraphicsDevice(long address);

    /**
     * * D3D device pointer or OpenGl context handle
     * *
     * * Only applicable when ctx is NULL, thus forcing a new context to be
     * * created.  In that case, the created context will be bound to this
     * * graphics device.
     */
    public void setGraphicsDevice(NativeObject value) {
        checkNotNull();
        _setGraphicsDevice(address, value.getAddress());
    }
    private static native void _setGraphicsDevice(long address, long value);

    /**
     * * Application-specific GUID
     * *
     * * If your application employs PhysX modules that use CUDA you need to use a GUID 
     * * so that patches for new architectures can be released for your game.You can obtain a GUID for your 
     * * application from Nvidia.
     */
    public String getAppGUID() {
        checkNotNull();
        return _getAppGUID(address);
    }
    private static native String _getAppGUID(long address);

    /**
     * * Application-specific GUID
     * *
     * * If your application employs PhysX modules that use CUDA you need to use a GUID 
     * * so that patches for new architectures can be released for your game.You can obtain a GUID for your 
     * * application from Nvidia.
     */
    public void setAppGUID(String value) {
        checkNotNull();
        _setAppGUID(address, value);
    }
    private static native void _setAppGUID(long address, String value);

}
