package physx.cooking;

/**
 * Flags which describe the format and behavior of a convex mesh.
 */
public enum PxConvexFlagEnum {

    /**
     * Denotes the use of 16-bit vertex indices in PxConvexMeshDesc::triangles or PxConvexMeshDesc::polygons.
     * (otherwise, 32-bit indices are assumed)
     * <b>See also:</b> #PxConvexMeshDesc.indices
     */
    e16_BIT_INDICES(gete16_BIT_INDICES()),
    /**
     * Automatically recomputes the hull from the vertices. If this flag is not set, you must provide the entire geometry manually.
     * <p>
     * <b>Note:</b> There are two different algorithms for hull computation, please see PxConvexMeshCookingType. 
     * <p>
     * <b>See also:</b> PxConvexMeshCookingType
     */
    eCOMPUTE_CONVEX(geteCOMPUTE_CONVEX()),
    /**
     * Checks and removes almost zero-area triangles during convex hull computation. 
     * The rejected area size is specified in PxCookingParams::areaTestEpsilon
     * <p>
     * <b>Note:</b> This flag is only used in combination with eCOMPUTE_CONVEX.
     * <p>
     * <b>See also:</b> PxCookingParams PxCookingParams::areaTestEpsilon
     */
    eCHECK_ZERO_AREA_TRIANGLES(geteCHECK_ZERO_AREA_TRIANGLES()),
    /**
     * Quantizes the input vertices using the k-means clustering
     * <p>
     * <b>Note:</b> The input vertices are quantized to PxConvexMeshDesc::quantizedCount
     * see http://en.wikipedia.org/wiki/K-means_clustering
     */
    eQUANTIZE_INPUT(geteQUANTIZE_INPUT()),
    /**
     * Disables the convex mesh validation to speed-up hull creation. Please use separate validation
     * function in checked/debug builds. Creating a convex mesh with invalid input data without prior validation
     * may result in undefined behavior. 
     * <p>
     * <b>See also:</b> PxCooking::validateConvexMesh
     */
    eDISABLE_MESH_VALIDATION(geteDISABLE_MESH_VALIDATION()),
    /**
     * Enables plane shifting vertex limit algorithm.
     * <p>
     * Plane shifting is an alternative algorithm for the case when the computed hull has more vertices 
     * than the specified vertex limit.
     * <p>
     * The default algorithm computes the full hull, and an OBB around the input vertices. This OBB is then sliced
     * with the hull planes until the vertex limit is reached.The default algorithm requires the vertex limit 
     * to be set to at least 8, and typically produces results that are much better quality than are produced 
     * by plane shifting.
     * <p>
     * When plane shifting is enabled, the hull computation stops when vertex limit is reached. The hull planes
     * are then shifted to contain all input vertices, and the new plane intersection points are then used to 
     * generate the final hull with the given vertex limit.Plane shifting may produce sharp edges to vertices 
     * very far away from the input cloud, and does not guarantee that all input vertices are inside the resulting
     * hull.However, it can be used with a vertex limit as low as 4.
     */
    ePLANE_SHIFTING(getePLANE_SHIFTING()),
    /**
     * Inertia tensor computation is faster using SIMD code, but the precision is lower, which may result 
     * in incorrect inertia for very thin hulls.
     */
    eFAST_INERTIA_COMPUTATION(geteFAST_INERTIA_COMPUTATION()),
    eGPU_COMPATIBLE(geteGPU_COMPATIBLE()),
    /**
     * Convex hull input vertices are shifted to be around origin to provide better computation stability.
     * It is recommended to provide input vertices around the origin, otherwise use this flag to improve 
     * numerical stability.
     * <b>Note:</b> Is used only with eCOMPUTE_CONVEX flag.
     */
    eSHIFT_VERTICES(geteSHIFT_VERTICES());
    public final int value;
    
    PxConvexFlagEnum(int value) {
        this.value = value;
    }

    private static native int _gete16_BIT_INDICES();
    private static int gete16_BIT_INDICES() {
        de.fabmax.physxjni.Loader.load();
        return _gete16_BIT_INDICES();
    }

    private static native int _geteCOMPUTE_CONVEX();
    private static int geteCOMPUTE_CONVEX() {
        de.fabmax.physxjni.Loader.load();
        return _geteCOMPUTE_CONVEX();
    }

    private static native int _geteCHECK_ZERO_AREA_TRIANGLES();
    private static int geteCHECK_ZERO_AREA_TRIANGLES() {
        de.fabmax.physxjni.Loader.load();
        return _geteCHECK_ZERO_AREA_TRIANGLES();
    }

    private static native int _geteQUANTIZE_INPUT();
    private static int geteQUANTIZE_INPUT() {
        de.fabmax.physxjni.Loader.load();
        return _geteQUANTIZE_INPUT();
    }

    private static native int _geteDISABLE_MESH_VALIDATION();
    private static int geteDISABLE_MESH_VALIDATION() {
        de.fabmax.physxjni.Loader.load();
        return _geteDISABLE_MESH_VALIDATION();
    }

    private static native int _getePLANE_SHIFTING();
    private static int getePLANE_SHIFTING() {
        de.fabmax.physxjni.Loader.load();
        return _getePLANE_SHIFTING();
    }

    private static native int _geteFAST_INERTIA_COMPUTATION();
    private static int geteFAST_INERTIA_COMPUTATION() {
        de.fabmax.physxjni.Loader.load();
        return _geteFAST_INERTIA_COMPUTATION();
    }

    private static native int _geteGPU_COMPATIBLE();
    private static int geteGPU_COMPATIBLE() {
        de.fabmax.physxjni.Loader.load();
        return _geteGPU_COMPATIBLE();
    }

    private static native int _geteSHIFT_VERTICES();
    private static int geteSHIFT_VERTICES() {
        de.fabmax.physxjni.Loader.load();
        return _geteSHIFT_VERTICES();
    }

    public static PxConvexFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxConvexFlagEnum: " + value);
    }

}
