package physx.extensions;

import physx.common.PxTransform;
import physx.common.PxVec3;

/**
 *  A D6 joint is a general constraint between two actors.
 * <p>
 *  It allows the application to individually define the linear and rotational degrees of freedom, 
 *  and also to configure a variety of limits and driven degrees of freedom.
 * <p>
 *  By default all degrees of freedom are locked. So to create a prismatic joint with free motion 
 *  along the x-axis:
 * <p>
 *  \code 
 *     ...
 *     joint-&gt;setMotion(PxD6Axis::eX, PxD6JointMotion::eFREE);
 *      ...
 *  \endcode
 * <p>
 *  Or a Revolute joint with motion free allowed around the x-axis:
 * <p>
 *  \code
 *     ... 
 *  joint-&gt;setMotion(PxD6Axis::eTWIST, PxD6JointMotion::eFREE);
 *     ...
 *  \endcode
 * <p>
 *  Degrees of freedom may also be set to limited instead of locked.
 * <p>
 *  There are two different kinds of linear limits available. The first kind is a single limit value
 *  for all linear degrees of freedom, which may act as a linear, circular, or spherical limit depending
 *  on which degrees of freedom are limited. This is similar to a distance limit. Then, the second kind
 *  supports a pair of limit values for each linear axis, which can be used to implement a traditional
 *  prismatic joint for example.
 * <p>
 *  If the twist degree of freedom is limited, is supports upper and lower limits. The two swing degrees
 *  of freedom are limited with a cone limit.
 * @see PxJoint
 */
public class PxD6Joint extends PxJoint {

    protected PxD6Joint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxD6Joint wrapPointer(long address) {
        return address != 0L ? new PxD6Joint(address) : null;
    }
    
    public static PxD6Joint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxD6Joint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Set the motion type around the specified axis.
     * <p>
     * Each axis may independently specify that the degree of freedom is locked (blocking relative movement
     * along or around this axis), limited by the corresponding limit, or free.
     * @param axis the axis around which motion is specified
     * @param type the motion type around the specified axis
     * <p>
     * <b>Default:</b> all degrees of freedom are locked
     * @see #getMotion
     */
    public void setMotion(PxD6AxisEnum axis, PxD6MotionEnum type) {
        checkNotNull();
        _setMotion(address, axis.value, type.value);
    }
    private static native void _setMotion(long address, int axis, int type);

    /**
     * Get the motion type around the specified axis.
     * @see #setMotion
     * @param axis the degree of freedom around which the motion type is specified
     * @return the motion type around the specified axis
     */
    public PxD6MotionEnum getMotion(PxD6AxisEnum axis) {
        checkNotNull();
        return PxD6MotionEnum.forValue(_getMotion(address, axis.value));
    }
    private static native int _getMotion(long address, int axis);

    /**
     * get the twist angle of the joint, in the range (-2*Pi, 2*Pi]
     */
    public float getTwistAngle() {
        checkNotNull();
        return _getTwistAngle(address);
    }
    private static native float _getTwistAngle(long address);

    /**
     * get the swing angle of the joint from the Y axis
     */
    public float getSwingYAngle() {
        checkNotNull();
        return _getSwingYAngle(address);
    }
    private static native float _getSwingYAngle(long address);

    /**
     * get the swing angle of the joint from the Z axis
     */
    public float getSwingZAngle() {
        checkNotNull();
        return _getSwingZAngle(address);
    }
    private static native float _getSwingZAngle(long address);

    /**
     * Set the distance limit for the joint. 
     * <p>
     * A single limit constraints all linear limited degrees of freedom, forming a linear, circular 
     * or spherical constraint on motion depending on the number of limited degrees. This is similar
     * to a distance limit.
     * @param limit the distance limit structure
     * @see PxJointLinearLimit
     */
    public void setDistanceLimit(PxJointLinearLimit limit) {
        checkNotNull();
        _setDistanceLimit(address, limit.getAddress());
    }
    private static native void _setDistanceLimit(long address, long limit);

    /**
     * Set the linear limit for a given linear axis. 
     * <p>
     * This function extends the previous setDistanceLimit call with the following features:
     * - there can be a different limit for each linear axis
     * - each limit is defined by two values, i.e. it can now be asymmetric
     * <p>
     * This can be used to create prismatic joints similar to PxPrismaticJoint, or point-in-quad joints,
     * or point-in-box joints.
     * @param axis  The limited linear axis (must be PxD6Axis::eX, PxD6Axis::eY or PxD6Axis::eZ)
     * @param limit The linear limit pair structure
     */
    public void setLinearLimit(PxD6AxisEnum axis, PxJointLinearLimitPair limit) {
        checkNotNull();
        _setLinearLimit(address, axis.value, limit.getAddress());
    }
    private static native void _setLinearLimit(long address, int axis, long limit);

    /**
     * Set the twist limit for the joint. 
     * <p>
     * The twist limit controls the range of motion around the twist axis. 
     * <p>
     * The limit angle range is (-2*Pi, 2*Pi).
     * @param limit the twist limit structure
     * @see PxJointAngularLimitPair
     */
    public void setTwistLimit(PxJointAngularLimitPair limit) {
        checkNotNull();
        _setTwistLimit(address, limit.getAddress());
    }
    private static native void _setTwistLimit(long address, long limit);

    /**
     * Set the swing cone limit for the joint. 
     * <p>
     * The cone limit is used if either or both swing axes are limited. The extents are 
     * symmetrical and measured in the frame of the parent. If only one swing degree of freedom 
     * is limited, the corresponding value from the cone limit defines the limit range.
     * @param limit the cone limit structure
     * @see PxJointLimitCone
     */
    public void setSwingLimit(PxJointLimitCone limit) {
        checkNotNull();
        _setSwingLimit(address, limit.getAddress());
    }
    private static native void _setSwingLimit(long address, long limit);

    /**
     * Set a pyramidal swing limit for the joint. 
     * <p>
     * The pyramid limits will only be used in the following cases:
     * - both swing Y and Z are limited. The limit shape is then a pyramid.
     * - Y is limited and Z is locked, or vice versa. The limit shape is an asymmetric angular section, similar to
     * what is supported for the twist axis.
     * The remaining cases (Y limited and Z is free, or vice versa) are not supported.
     * @param limit the cone limit structure
     * @see PxJointLimitPyramid
     */
    public void setPyramidSwingLimit(PxJointLimitPyramid limit) {
        checkNotNull();
        _setPyramidSwingLimit(address, limit.getAddress());
    }
    private static native void _setPyramidSwingLimit(long address, long limit);

    /**
     * Set the drive parameters for the specified drive type.
     * @param index the type of drive being specified
     * @param drive the drive parameters
     * @see #getDrive
     * @see PxD6JointDrive
     * <p>
     * <b>Default</b> The default drive spring and damping values are zero, the force limit is zero, and no flags are set.
     */
    public void setDrive(PxD6DriveEnum index, PxD6JointDrive drive) {
        checkNotNull();
        _setDrive(address, index.value, drive.getAddress());
    }
    private static native void _setDrive(long address, int index, long drive);

    /**
     * Get the drive parameters for the specified drive type. 
     * @param index the specified drive type
     * @see #setDrive
     * @see PxD6JointDrive
     */
    public PxD6JointDrive getDrive(PxD6DriveEnum index) {
        checkNotNull();
        return PxD6JointDrive.wrapPointer(_getDrive(address, index.value));
    }
    private static native long _getDrive(long address, int index);

    /**
     * Set the drive goal pose 
     * <p>
     * The goal is relative to the constraint frame of actor[0]
     * <p>
     * <b>Default</b> the identity transform
     * @param pose The goal drive pose if positional drive is in use. 
     * wake counters to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * @see #setDrivePosition
     */
    public void setDrivePosition(PxTransform pose) {
        checkNotNull();
        _setDrivePosition(address, pose.getAddress());
    }
    private static native void _setDrivePosition(long address, long pose);

    /**
     * Set the drive goal pose 
     * <p>
     * The goal is relative to the constraint frame of actor[0]
     * <p>
     * <b>Default</b> the identity transform
     * @param pose The goal drive pose if positional drive is in use. 
     * @param autowake If true and the attached actors are in a scene, this call wakes them up and increases their
     * wake counters to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * @see #setDrivePosition
     */
    public void setDrivePosition(PxTransform pose, boolean autowake) {
        checkNotNull();
        _setDrivePosition(address, pose.getAddress(), autowake);
    }
    private static native void _setDrivePosition(long address, long pose, boolean autowake);

    /**
     * Get the drive goal pose.
     * @see #getDrivePosition
     */
    public PxTransform getDrivePosition() {
        checkNotNull();
        return PxTransform.wrapPointer(_getDrivePosition(address));
    }
    private static native long _getDrivePosition(long address);

    /**
     * Set the target goal velocity for drive.
     * <p>
     * The velocity is measured in the constraint frame of actor[0]
     * @param linear The goal velocity for linear drive
     * @param angular The goal velocity for angular drive
     * wake counters to #PxSceneDesc::wakeCounterResetValue if the counter value is below the reset value.
     * @see #getDriveVelocity
     */
    public void setDriveVelocity(PxVec3 linear, PxVec3 angular) {
        checkNotNull();
        _setDriveVelocity(address, linear.getAddress(), angular.getAddress());
    }
    private static native void _setDriveVelocity(long address, long linear, long angular);

    /**
     * Get the target goal velocity for joint drive.
     * @param linear The goal velocity for linear drive
     * @param angular The goal velocity for angular drive
     * @see #setDriveVelocity
     */
    public void getDriveVelocity(PxVec3 linear, PxVec3 angular) {
        checkNotNull();
        _getDriveVelocity(address, linear.getAddress(), angular.getAddress());
    }
    private static native void _getDriveVelocity(long address, long linear, long angular);

}
