package physx.extensions;


/**
 * A joint which behaves in a similar way to a ball and socket.
 * <p>
 *  A spherical joint removes all linear degrees of freedom from two objects.
 * <p>
 *  The position of the joint on each actor is specified by the origin of the body's joint frame.
 * <p>
 *  A spherical joint may have a cone limit, to restrict the motion to within a certain range. In
 *  addition, the bodies may be projected together if the distance between them exceeds a given threshold.
 * <p>
 *  Projection, drive and limits are activated by setting the appropriate flags on the joint.
 * @see PxJoint
 */
public class PxSphericalJoint extends PxJoint {

    protected PxSphericalJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSphericalJoint wrapPointer(long address) {
        return address != 0L ? new PxSphericalJoint(address) : null;
    }
    
    public static PxSphericalJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSphericalJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Get the limit cone.
     * @see PxJointLimitCone
     */
    public void setLimitCone(PxJointLimitCone limitCone) {
        checkNotNull();
        _setLimitCone(address, limitCone.getAddress());
    }
    private static native void _setLimitCone(long address, long limitCone);

    /**
     * get the swing angle of the joint from the Y axis
     */
    public float getSwingYAngle() {
        checkNotNull();
        return _getSwingYAngle(address);
    }
    private static native float _getSwingYAngle(long address);

    /**
     * get the swing angle of the joint from the Z axis
     */
    public float getSwingZAngle() {
        checkNotNull();
        return _getSwingZAngle(address);
    }
    private static native float _getSwingZAngle(long address);

    /**
     * Set the flags specific to the Spherical Joint.
     * <p>
     * <b>Default</b> PxSphericalJointFlags(0)
     * @param flags The joint flags.
     */
    public void setSphericalJointFlags(PxSphericalJointFlags flags) {
        checkNotNull();
        _setSphericalJointFlags(address, flags.getAddress());
    }
    private static native void _setSphericalJointFlags(long address, long flags);

    /**
     * Set a single flag specific to a Spherical Joint to true or false.
     * @param flag The flag to set or clear.
     * @param value the value to which to set the flag
     */
    public void setSphericalJointFlag(PxSphericalJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setSphericalJointFlag(address, flag.value, value);
    }
    private static native void _setSphericalJointFlag(long address, int flag, boolean value);

    /**
     * Get the flags specific to the Spherical Joint.
     * @return the joint flags
     */
    public PxSphericalJointFlags getSphericalJointFlags() {
        checkNotNull();
        return PxSphericalJointFlags.wrapPointer(_getSphericalJointFlags(address));
    }
    private static native long _getSphericalJointFlags(long address);

}
