package physx.particles;

import physx.NativeObject;
import physx.PlatformChecks;
import physx.common.PxVec4;
import physx.support.PxU32Ptr;

/**
 * Helper class to manage PxParticleClothDesc buffers used for communicating particle based cloths to PxParticleClothBuffer.
 */
public class PxParticleClothBufferHelper extends NativeObject {

    static {
        PlatformChecks.requirePlatform(3, "physx.particles.PxParticleClothBufferHelper");
    }

    protected PxParticleClothBufferHelper() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxParticleClothBufferHelper wrapPointer(long address) {
        return address != 0L ? new PxParticleClothBufferHelper(address) : null;
    }
    
    public static PxParticleClothBufferHelper arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxParticleClothBufferHelper(long address) {
        super(address);
    }

    // Functions

    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * @return The maximum number of cloths this PxParticleClothBufferHelper can hold.
     */
    public int getMaxCloths() {
        checkNotNull();
        return _getMaxCloths(address);
    }
    private static native int _getMaxCloths(long address);

    /**
     * @return The current number of cloths in this PxParticleClothBufferHelper.
     */
    public int getNumCloths() {
        checkNotNull();
        return _getNumCloths(address);
    }
    private static native int _getNumCloths(long address);

    /**
     * @return The maximum number of springs this PxParticleClothBufferHelper can hold.
     */
    public int getMaxSprings() {
        checkNotNull();
        return _getMaxSprings(address);
    }
    private static native int _getMaxSprings(long address);

    /**
     * @return The current number of springs in this PxParticleClothBufferHelper.
     */
    public int getNumSprings() {
        checkNotNull();
        return _getNumSprings(address);
    }
    private static native int _getNumSprings(long address);

    /**
     * @return The maximum number of triangles this PxParticleClothBufferHelper can hold.
     */
    public int getMaxTriangles() {
        checkNotNull();
        return _getMaxTriangles(address);
    }
    private static native int _getMaxTriangles(long address);

    /**
     * @return The current number of triangles in this PxParticleClothBufferHelper.
     */
    public int getNumTriangles() {
        checkNotNull();
        return _getNumTriangles(address);
    }
    private static native int _getNumTriangles(long address);

    /**
     * @return The maximum number of particles this PxParticleClothBufferHelper can hold.
     */
    public int getMaxParticles() {
        checkNotNull();
        return _getMaxParticles(address);
    }
    private static native int _getMaxParticles(long address);

    /**
     * @return The current number of particles in this PxParticleClothBufferHelper.
     */
    public int getNumParticles() {
        checkNotNull();
        return _getNumParticles(address);
    }
    private static native int _getNumParticles(long address);

    /**
     * Adds a PxParticleCloth to this PxParticleClothBufferHelper instance.
     * @param particleCloth The PxParticleCloth to be added.
     * @param triangles A pointer to the triangles
     * @param numTriangles The number of triangles
     * @param springs A pointer to the springs
     * @param numSprings The number of springs
     * @param restPositions A pointer to the particle rest positions
     * @param numParticles The number of particles in this cloth
     * @see PxParticleCloth
     * @see PxParticleSpring
     */
    public void addCloth(PxParticleCloth particleCloth, PxU32Ptr triangles, int numTriangles, PxParticleSpring springs, int numSprings, PxVec4 restPositions, int numParticles) {
        checkNotNull();
        _addCloth(address, particleCloth.getAddress(), triangles.getAddress(), numTriangles, springs.getAddress(), numSprings, restPositions.getAddress(), numParticles);
    }
    private static native void _addCloth(long address, long particleCloth, long triangles, int numTriangles, long springs, int numSprings, long restPositions, int numParticles);

    /**
     * Adds a cloth to this PxParticleClothBufferHelper instance.
     * <p>
     * Adds a cloth to this PxParticleClothBufferHelper instance. With this method the relevant parameters for inflatable simulation
     * (restVolume, pressure) can be set directly.
     * @param blendScale This should be 1.f / (numPartitions + 1) if the springs are partitioned by the user. Otherwise this will be set during spring partitioning.
     * @param restVolume The rest volume of the inflatable
     * @param pressure The pressure of the inflatable. The target inflatable volume is defined as restVolume * pressure. Setting this to &gt; 0.0 will enable inflatable simulation.
     * @param triangles A pointer to the triangles
     * @param numTriangles The number of triangles
     * @param springs A pointer to the springs
     * @param numSprings The number of springs
     * @param restPositions A pointer to the particle rest positions
     * @param numParticles The number of particles in this cloth
     * @see PxParticleSpring
     */
    public void addCloth(float blendScale, float restVolume, float pressure, PxU32Ptr triangles, int numTriangles, PxParticleSpring springs, int numSprings, PxVec4 restPositions, int numParticles) {
        checkNotNull();
        _addCloth(address, blendScale, restVolume, pressure, triangles.getAddress(), numTriangles, springs.getAddress(), numSprings, restPositions.getAddress(), numParticles);
    }
    private static native void _addCloth(long address, float blendScale, float restVolume, float pressure, long triangles, int numTriangles, long springs, int numSprings, long restPositions, int numParticles);

    /**
     * Returns a PxParticleClothDesc for this PxParticleClothBufferHelper instance to be used for spring partitioning.
     * @return the PxParticleClothDesc.
     */
    public PxParticleClothDesc getParticleClothDesc() {
        checkNotNull();
        return PxParticleClothDesc.wrapPointer(_getParticleClothDesc(address));
    }
    private static native long _getParticleClothDesc(long address);

}
