package physx.physics;


/**
 * A component of an articulation that represents a rigid body.
 * <p>
 * Articulation links have a restricted subset of the functionality of a PxRigidDynamic:
 * - They may not be kinematic, and do not support contact-force thresholds.
 * - Their velocity or global pose cannot be set directly, but must be set via the articulation-root and joint positions/velocities.
 * - Sleep state and solver iteration counts are properties of the entire articulation rather than the individual links.
 * @see PxRigidBody
 */
public class PxArticulationLink extends PxRigidBody {

    protected PxArticulationLink() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationLink wrapPointer(long address) {
        return address != 0L ? new PxArticulationLink(address) : null;
    }
    
    public static PxArticulationLink arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationLink(long address) {
        super(address);
    }

    // Functions

    /**
     * Gets the articulation that the link is a part of.
     * @return The articulation.
     * @see PxArticulationReducedCoordinate
     */
    public PxArticulationReducedCoordinate getArticulation() {
        checkNotNull();
        return PxArticulationReducedCoordinate.wrapPointer(_getArticulation(address));
    }
    private static native long _getArticulation(long address);

    /**
     * Gets the joint which connects this link to its parent.
     * @return The joint connecting the link to the parent. NULL for the root link.
     * @see PxArticulationJointReducedCoordinate
     */
    public PxArticulationJointReducedCoordinate getInboundJoint() {
        checkNotNull();
        return PxArticulationJointReducedCoordinate.wrapPointer(_getInboundJoint(address));
    }
    private static native long _getInboundJoint(long address);

    /**
     * Gets the number of degrees of freedom of the joint which connects this link to its parent.
     * <p>
     * - The root link DOF-count is defined to be 0 regardless of PxArticulationFlag::eFIX_BASE.
     * - The return value is only valid for articulations that are in a scene.
     * @return The number of degrees of freedom, or 0xFFFFFFFF if the articulation is not in a scene.
     * @see PxArticulationJointReducedCoordinate
     */
    public int getInboundJointDof() {
        checkNotNull();
        return _getInboundJointDof(address);
    }
    private static native int _getInboundJointDof(long address);

    /**
     * Gets the number of child links.
     * @return The number of child links.
     */
    public int getNbChildren() {
        checkNotNull();
        return _getNbChildren(address);
    }
    private static native int _getNbChildren(long address);

    /**
     * Gets the low-level link index that may be used to index into members of PxArticulationCache.
     * <p>
     * The return value is only valid for articulations that are in a scene.
     * @return The low-level index, or 0xFFFFFFFF if the articulation is not in a scene.
     * @see PxArticulationCache
     */
    public int getLinkIndex() {
        checkNotNull();
        return _getLinkIndex(address);
    }
    private static native int _getLinkIndex(long address);

    /**
     * Set the constraint-force-mixing scale term.
     * <p>
     * The cfm scale term is a stabilization term that helps avoid instabilities with over-constrained
     * configurations. It should be a small value that is multiplied by 1/mass internally to produce
     * an additional bias added to the unit response term in the solver.
     * @param cfm The constraint-force-mixing scale term.
     * <p>
     * <b>Default:</b> 0.025
     * <b>Range:</b> [0, 1]
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running.
     * @see #getCfmScale
     */
    public void setCfmScale(float cfm) {
        checkNotNull();
        _setCfmScale(address, cfm);
    }
    private static native void _setCfmScale(long address, float cfm);

    /**
     * Get the constraint-force-mixing scale term.
     * @return The constraint-force-mixing scale term.
     * @see #setCfmScale
     */
    public float getCfmScale() {
        checkNotNull();
        return _getCfmScale(address);
    }
    private static native float _getCfmScale(long address);

}
