package physx.physics;

/**
 * Enumeration that determines the way in which two material properties will be combined to yield a friction or restitution coefficient for a collision.
 * <p>
 * When two actors come in contact with each other, they each have materials with various coefficients, but we only need a single set of coefficients for the pair.
 * <p>
 * Physics doesn't have any inherent combinations because the coefficients are determined empirically on a case by case
 * basis. However, simulating this with a pairwise lookup table is often impractical.
 * <p>
 * For this reason the following combine behaviors are available:
 * <p>
 * eAVERAGE
 * eMIN
 * eMULTIPLY
 * eMAX
 * <p>
 * The effective combine mode for the pair is maximum(material0.combineMode, material1.combineMode).
 * <p>
 * <b>See also:</b> PxMaterial.setFrictionCombineMode() PxMaterial.getFrictionCombineMode() PxMaterial.setRestitutionCombineMode() PxMaterial.getFrictionCombineMode()
 */
public enum PxCombineModeEnum {

    /**
     * Average: (a + b)/2
     */
    eAVERAGE(geteAVERAGE()),
    /**
     * Minimum: minimum(a,b)
     */
    eMIN(geteMIN()),
    /**
     * Multiply: a*b
     */
    eMULTIPLY(geteMULTIPLY()),
    /**
     * Maximum: maximum(a,b)
     */
    eMAX(geteMAX());
    public final int value;
    
    PxCombineModeEnum(int value) {
        this.value = value;
    }

    private static native int _geteAVERAGE();
    private static int geteAVERAGE() {
        de.fabmax.physxjni.Loader.load();
        return _geteAVERAGE();
    }

    private static native int _geteMIN();
    private static int geteMIN() {
        de.fabmax.physxjni.Loader.load();
        return _geteMIN();
    }

    private static native int _geteMULTIPLY();
    private static int geteMULTIPLY() {
        de.fabmax.physxjni.Loader.load();
        return _geteMULTIPLY();
    }

    private static native int _geteMAX();
    private static int geteMAX() {
        de.fabmax.physxjni.Loader.load();
        return _geteMAX();
    }

    public static PxCombineModeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxCombineModeEnum: " + value);
    }

}
