package physx.physics;

/**
 * Enum for selecting the friction algorithm used for simulation.
 * <p>
 * #PxFrictionType::ePATCH selects the patch friction model which typically leads to the most stable results at low solver iteration counts and is also quite inexpensive, as it uses only
 * up to four scalar solver constraints per pair of touching objects.  The patch friction model is the same basic strong friction algorithm as PhysX 3.2 and before.  
 * <p>
 * #PxFrictionType::eTWO_DIRECTIONAL is identical to the one directional model, but it applies friction in both tangent directions simultaneously.  This hurts convergence a bit so it 
 * requires more solver iterations, but is more accurate.  Like the one directional model, it is applied at every contact point, which makes it potentially more expensive
 * than patch friction for scenarios with many contact points.
 * <p>
 * #PxFrictionType::eFRICTION_COUNT is the total numer of friction models supported by the SDK.
 */
public enum PxFrictionTypeEnum {

    /**
     * Select default patch-friction model.
     */
    ePATCH(getePATCH()),
    eONE_DIRECTIONAL(geteONE_DIRECTIONAL()),
    /**
     * Select two directional per-contact friction model.
     */
    eTWO_DIRECTIONAL(geteTWO_DIRECTIONAL()),
    eFRICTION_COUNT(geteFRICTION_COUNT());
    public final int value;
    
    PxFrictionTypeEnum(int value) {
        this.value = value;
    }

    private static native int _getePATCH();
    private static int getePATCH() {
        de.fabmax.physxjni.Loader.load();
        return _getePATCH();
    }

    private static native int _geteONE_DIRECTIONAL();
    private static int geteONE_DIRECTIONAL() {
        de.fabmax.physxjni.Loader.load();
        return _geteONE_DIRECTIONAL();
    }

    private static native int _geteTWO_DIRECTIONAL();
    private static int geteTWO_DIRECTIONAL() {
        de.fabmax.physxjni.Loader.load();
        return _geteTWO_DIRECTIONAL();
    }

    private static native int _geteFRICTION_COUNT();
    private static int geteFRICTION_COUNT() {
        de.fabmax.physxjni.Loader.load();
        return _geteFRICTION_COUNT();
    }

    public static PxFrictionTypeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxFrictionTypeEnum: " + value);
    }

}
