package physx.physics;

import physx.common.PxTransform;
import physx.common.PxVec3;

/**
 * PxRigidBody is a base class shared between dynamic rigid body objects.
 * @see PxRigidActor
 */
public class PxRigidBody extends PxRigidActor {

    protected PxRigidBody() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxRigidBody wrapPointer(long address) {
        return address != 0L ? new PxRigidBody(address) : null;
    }
    
    public static PxRigidBody arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxRigidBody(long address) {
        super(address);
    }

    // Functions

    /**
     * Sets the pose of the center of mass relative to the actor. 
     * <p>
     * <b>Note:</b> Changing this transform will not move the actor in the world!
     * <p>
     * <b>Note:</b> Setting an unrealistic center of mass which is a long way from the body can make it difficult for
     * the SDK to solve constraints. Perhaps leading to instability and jittering bodies.
     * <p>
     * <b>Note:</b> Changing this transform will not update the linear velocity reported by getLinearVelocity() to account
     * for the shift in center of mass. If the shift should be accounted for, the user should update the velocity
     * using setLinearVelocity().
     * <p>
     * <b>Default:</b> the identity transform
     * @param pose Mass frame offset transform relative to the actor frame. <b>Range:</b> rigid body transform.
     * @see #getCMassLocalPose
     * @see #getLinearVelocity
     */
    public void setCMassLocalPose(PxTransform pose) {
        checkNotNull();
        _setCMassLocalPose(address, pose.getAddress());
    }
    private static native void _setCMassLocalPose(long address, long pose);

    /**
     * Retrieves the center of mass pose relative to the actor frame.
     * @return The center of mass pose relative to the actor frame.
     * @see #setCMassLocalPose
     */
    public PxTransform getCMassLocalPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getCMassLocalPose(address));
    }
    private static native long _getCMassLocalPose(long address);

    /**
     * Sets the mass of a dynamic actor.
     * <p>
     * The mass must be non-negative.
     * <p>
     * setMass() does not update the inertial properties of the body, to change the inertia tensor
     * use setMassSpaceInertiaTensor() or the PhysX extensions method #PxRigidBodyExt::updateMassAndInertia().
     * <p>
     * <b>Note:</b> A value of 0 is interpreted as infinite mass.
     * <b>Note:</b> Values of 0 are not permitted for instances of PxArticulationLink but are permitted for instances of PxRigidDynamic. 
     * <p>
     * <b>Default:</b> 1.0
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param mass New mass value for the actor. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getMass
     * @see #setMassSpaceInertiaTensor
     */
    public void setMass(float mass) {
        checkNotNull();
        _setMass(address, mass);
    }
    private static native void _setMass(long address, float mass);

    /**
     * Retrieves the mass of the actor.
     * <p>
     * <b>Note:</b> A value of 0 is interpreted as infinite mass.
     * @return The mass of this actor.
     * @see #setMass
     * @see #setMassSpaceInertiaTensor
     */
    public float getMass() {
        checkNotNull();
        return _getMass(address);
    }
    private static native float _getMass(long address);

    /**
     * Retrieves the inverse mass of the actor.
     * @return The inverse mass of this actor.
     * @see #setMass
     * @see #setMassSpaceInertiaTensor
     */
    public float getInvMass() {
        checkNotNull();
        return _getInvMass(address);
    }
    private static native float _getInvMass(long address);

    /**
     * Sets the inertia tensor, using a parameter specified in mass space coordinates.
     * <p>
     * Note that such matrices are diagonal -- the passed vector is the diagonal.
     * <p>
     * If you have a non diagonal world/actor space inertia tensor(3x3 matrix). Then you need to
     * diagonalize it and set an appropriate mass space transform. See #setCMassLocalPose().
     * <p>
     * The inertia tensor elements must be non-negative.
     * <p>
     * <b>Note:</b> A value of 0 in an element is interpreted as infinite inertia along that axis.
     * <b>Note:</b> Values of 0 are not permitted for instances of PxArticulationLink but are permitted for instances of PxRigidDynamic. 
     * <p>
     * <b>Default:</b> (1.0, 1.0, 1.0)
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param m New mass space inertia tensor for the actor.
     * @see #setMass
     * @see #setCMassLocalPose
     */
    public void setMassSpaceInertiaTensor(PxVec3 m) {
        checkNotNull();
        _setMassSpaceInertiaTensor(address, m.getAddress());
    }
    private static native void _setMassSpaceInertiaTensor(long address, long m);

    /**
     *  Retrieves the diagonal inertia tensor of the actor relative to the mass coordinate frame.
     * <p>
     * This method retrieves a mass frame inertia vector.
     * @return The mass space inertia tensor of this actor.
     * <p>
     * <b>Note:</b> A value of 0 in an element is interpreted as infinite inertia along that axis.
     * @see #setMassSpaceInertiaTensor
     * @see #setMass
     * @see #setCMassLocalPose
     */
    public PxVec3 getMassSpaceInertiaTensor() {
        checkNotNull();
        return PxVec3.wrapPointer(_getMassSpaceInertiaTensor(address));
    }
    private static native long _getMassSpaceInertiaTensor(long address);

    /**
     *  Retrieves the diagonal inverse inertia tensor of the actor relative to the mass coordinate frame.
     * <p>
     * This method retrieves a mass frame inverse inertia vector.
     * <p>
     * <b>Note:</b> A value of 0 in an element is interpreted as infinite inertia along that axis.
     * @return The mass space inverse inertia tensor of this actor.
     * @see #setMassSpaceInertiaTensor
     * @see #setMass
     * @see #setCMassLocalPose
     */
    public PxVec3 getMassSpaceInvInertiaTensor() {
        checkNotNull();
        return PxVec3.wrapPointer(_getMassSpaceInvInertiaTensor(address));
    }
    private static native long _getMassSpaceInvInertiaTensor(long address);

    /**
     * Sets the linear damping coefficient.
     * <p>
     * Zero represents no damping. The damping coefficient must be nonnegative.
     * <p>
     * <b>Default:</b> 0.05 for PxArticulationLink, 0.0 for PxRigidDynamic 
     * @param linDamp Linear damping coefficient. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getLinearDamping
     * @see #setAngularDamping
     */
    public void setLinearDamping(float linDamp) {
        checkNotNull();
        _setLinearDamping(address, linDamp);
    }
    private static native void _setLinearDamping(long address, float linDamp);

    /**
     * Retrieves the linear damping coefficient.
     * @return The linear damping coefficient associated with this actor.
     * @see #setLinearDamping
     * @see #getAngularDamping
     */
    public float getLinearDamping() {
        checkNotNull();
        return _getLinearDamping(address);
    }
    private static native float _getLinearDamping(long address);

    /**
     * Sets the angular damping coefficient.
     * <p>
     * Zero represents no damping.
     * <p>
     * The angular damping coefficient must be nonnegative.
     * <p>
     * <b>Default:</b> 0.05
     * @param angDamp Angular damping coefficient. <b>Range:</b> [0, PX_MAX_F32)
     * @see #getAngularDamping
     * @see #setLinearDamping
     */
    public void setAngularDamping(float angDamp) {
        checkNotNull();
        _setAngularDamping(address, angDamp);
    }
    private static native void _setAngularDamping(long address, float angDamp);

    /**
     * Retrieves the angular damping coefficient.
     * @return The angular damping coefficient associated with this actor.
     * @see #setAngularDamping
     * @see #getLinearDamping
     */
    public float getAngularDamping() {
        checkNotNull();
        return _getAngularDamping(address);
    }
    private static native float _getAngularDamping(long address);

    /**
     * Retrieves the linear velocity of an actor.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * <p>
     * <b>Note:</b> The linear velocity is reported with respect to the rigid body's center of mass and not the actor frame origin.
     * @return The linear velocity of the actor.
     * @see PxRigidDynamic#setLinearVelocity
     * @see #getAngularVelocity
     */
    public PxVec3 getLinearVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getLinearVelocity(address));
    }
    private static native long _getLinearVelocity(long address);

    /**
     * Retrieves the angular velocity of the actor.
     * <p>
     * <b>Note:</b> It is not allowed to use this method while the simulation is running (except during PxScene::collide(),
     * in PxContactModifyCallback or in contact report callbacks).
     * @return The angular velocity of the actor.
     * @see PxRigidDynamic#setAngularVelocity
     * @see #getLinearVelocity
     */
    public PxVec3 getAngularVelocity() {
        checkNotNull();
        return PxVec3.wrapPointer(_getAngularVelocity(address));
    }
    private static native long _getAngularVelocity(long address);

    /**
     * Lets you set the maximum linear velocity permitted for this actor.
     * <p>
     * With this function, you can set the  maximum linear velocity permitted for this rigid body.
     * Higher linear velocities are clamped to this value.
     * <p>
     * Note: The linear velocity is clamped to the set value <i>before</i> the solver, which means that
     * the limit may still be momentarily exceeded.
     * <p>
     * <b>Default:</b> 100*PxTolerancesScale::length for PxArticulationLink, 1e^16 for PxRigidDynamic
     * @param maxLinVel Max allowable linear velocity for actor. <b>Range:</b> [0, 1e^16)
     * @see #getMaxAngularVelocity
     */
    public void setMaxLinearVelocity(float maxLinVel) {
        checkNotNull();
        _setMaxLinearVelocity(address, maxLinVel);
    }
    private static native void _setMaxLinearVelocity(long address, float maxLinVel);

    /**
     * Retrieves the maximum angular velocity permitted for this actor.
     * @return The maximum allowed angular velocity for this actor.
     * @see #setMaxLinearVelocity
     */
    public float getMaxLinearVelocity() {
        checkNotNull();
        return _getMaxLinearVelocity(address);
    }
    private static native float _getMaxLinearVelocity(long address);

    /**
     * Lets you set the maximum angular velocity permitted for this actor.
     * <p>
     * For various internal computations, very quickly rotating actors introduce error
     * into the simulation, which leads to undesired results.
     * <p>
     * With this function, you can set the  maximum angular velocity permitted for this rigid body.
     * Higher angular velocities are clamped to this value.
     * <p>
     * Note: The angular velocity is clamped to the set value <i>before</i> the solver, which means that
     * the limit may still be momentarily exceeded.
     * <p>
     * <b>Default:</b> 50.0 for PxArticulationLink, 100.0 for PxRigidDynamic
     * <p>
     * <b>Range:</b> [0, 1e^16)
     * @param maxAngVel Max allowable angular velocity for actor. 
     * @see #getMaxAngularVelocity
     */
    public void setMaxAngularVelocity(float maxAngVel) {
        checkNotNull();
        _setMaxAngularVelocity(address, maxAngVel);
    }
    private static native void _setMaxAngularVelocity(long address, float maxAngVel);

    /**
     * Retrieves the maximum angular velocity permitted for this actor.
     * @return The maximum allowed angular velocity for this actor.
     * @see #setMaxAngularVelocity
     */
    public float getMaxAngularVelocity() {
        checkNotNull();
        return _getMaxAngularVelocity(address);
    }
    private static native float _getMaxAngularVelocity(long address);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame to the actor at its center of mass.
     * <p>
     * <b>This will not induce a torque</b>.
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive.
     * <p>
     * Each actor has an acceleration and a velocity change accumulator which are directly modified using the modes PxForceMode::eACCELERATION 
     * and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE also modify these same 
     * accumulators and are just short hand for multiplying the vector parameter by inverse mass and then using PxForceMode::eACCELERATION and 
     * PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this is called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in linear velocity that 
     * will arise from the application of an impulsive force, where an impulsive force is applied force multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the force is non-zero.
     * @param force Force/Impulse to apply defined in the global frame.
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addTorque
     */
    public void addForce(PxVec3 force) {
        checkNotNull();
        _addForce(address, force.getAddress());
    }
    private static native void _addForce(long address, long force);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame to the actor at its center of mass.
     * <p>
     * <b>This will not induce a torque</b>.
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive.
     * <p>
     * Each actor has an acceleration and a velocity change accumulator which are directly modified using the modes PxForceMode::eACCELERATION 
     * and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE also modify these same 
     * accumulators and are just short hand for multiplying the vector parameter by inverse mass and then using PxForceMode::eACCELERATION and 
     * PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this is called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in linear velocity that 
     * will arise from the application of an impulsive force, where an impulsive force is applied force multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the force is non-zero.
     * @param force Force/Impulse to apply defined in the global frame.
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode)
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addTorque
     */
    public void addForce(PxVec3 force, PxForceModeEnum mode) {
        checkNotNull();
        _addForce(address, force.getAddress(), mode.value);
    }
    private static native void _addForce(long address, long force, int mode);

    /**
     * Applies a force (or impulse) defined in the global coordinate frame to the actor at its center of mass.
     * <p>
     * <b>This will not induce a torque</b>.
     * <p>
     * ::PxForceMode determines if the force is to be conventional or impulsive.
     * <p>
     * Each actor has an acceleration and a velocity change accumulator which are directly modified using the modes PxForceMode::eACCELERATION 
     * and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE also modify these same 
     * accumulators and are just short hand for multiplying the vector parameter by inverse mass and then using PxForceMode::eACCELERATION and 
     * PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this is called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in linear velocity that 
     * will arise from the application of an impulsive force, where an impulsive force is applied force multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the force is non-zero.
     * @param force Force/Impulse to apply defined in the global frame.
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode)
     * @param autowake Specify if the call should wake up the actor if it is currently asleep. If true and the current wake counter value
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addTorque
     */
    public void addForce(PxVec3 force, PxForceModeEnum mode, boolean autowake) {
        checkNotNull();
        _addForce(address, force.getAddress(), mode.value, autowake);
    }
    private static native void _addForce(long address, long force, int mode, boolean autowake);

    /**
     * Applies an impulsive torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the torque is to be conventional or impulsive.
     * <p>
     * Each actor has an angular acceleration and an angular velocity change accumulator which are directly modified using the modes 
     * PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE 
     * also modify these same accumulators and are just short hand for multiplying the vector parameter by inverse inertia and then 
     * using PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in angular velocity that 
     * will arise from the application of an impulsive torque, where an impulsive torque is an applied torque multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the torque is non-zero.
     * @param torque Torque to apply defined in the global frame. <b>Range:</b> torque vector
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addForce
     */
    public void addTorque(PxVec3 torque) {
        checkNotNull();
        _addTorque(address, torque.getAddress());
    }
    private static native void _addTorque(long address, long torque);

    /**
     * Applies an impulsive torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the torque is to be conventional or impulsive.
     * <p>
     * Each actor has an angular acceleration and an angular velocity change accumulator which are directly modified using the modes 
     * PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE 
     * also modify these same accumulators and are just short hand for multiplying the vector parameter by inverse inertia and then 
     * using PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in angular velocity that 
     * will arise from the application of an impulsive torque, where an impulsive torque is an applied torque multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the torque is non-zero.
     * @param torque Torque to apply defined in the global frame. <b>Range:</b> torque vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode).
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addForce
     */
    public void addTorque(PxVec3 torque, PxForceModeEnum mode) {
        checkNotNull();
        _addTorque(address, torque.getAddress(), mode.value);
    }
    private static native void _addTorque(long address, long torque, int mode);

    /**
     * Applies an impulsive torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the torque is to be conventional or impulsive.
     * <p>
     * Each actor has an angular acceleration and an angular velocity change accumulator which are directly modified using the modes 
     * PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE 
     * also modify these same accumulators and are just short hand for multiplying the vector parameter by inverse inertia and then 
     * using PxForceMode::eACCELERATION and PxForceMode::eVELOCITY_CHANGE respectively.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> if this called on an articulation link, only the link is updated, not the entire articulation.
     * <p>
     * <b>Note:</b> see #PxRigidBodyExt::computeVelocityDeltaFromImpulse for details of how to compute the change in angular velocity that 
     * will arise from the application of an impulsive torque, where an impulsive torque is an applied torque multiplied by a timestep.
     * <p>
     * <b>Sleeping:</b> This call wakes the actor if it is sleeping, and the autowake parameter is true (default) or the torque is non-zero.
     * @param torque Torque to apply defined in the global frame. <b>Range:</b> torque vector
     * @param mode The mode to use when applying the force/impulse(see #PxForceMode).
     * @param autowake Specify if the call should wake up the actor if it is currently asleep. If true and the current wake counter value
     * is smaller than #PxSceneDesc::wakeCounterResetValue it will get increased to the reset value.
     * @see #addForce
     */
    public void addTorque(PxVec3 torque, PxForceModeEnum mode, boolean autowake) {
        checkNotNull();
        _addTorque(address, torque.getAddress(), mode.value, autowake);
    }
    private static native void _addTorque(long address, long torque, int mode, boolean autowake);

    /**
     * Clears the accumulated forces (sets the accumulated force back to zero).
     * <p>
     * Each actor has an acceleration and a velocity change accumulator which are directly modified using the modes PxForceMode::eACCELERATION 
     * and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE also modify these same 
     * accumulators (see PxRigidBody::addForce() for details); therefore the effect of calling clearForce(PxForceMode::eFORCE) is equivalent to calling 
     * clearForce(PxForceMode::eACCELERATION), and the effect of calling clearForce(PxForceMode::eIMPULSE) is equivalent to calling 
     * clearForce(PxForceMode::eVELOCITY_CHANGE).
     * <p>
     * ::PxForceMode determines if the cleared force is to be conventional or impulsive.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param mode The mode to use when clearing the force/impulse(see #PxForceMode)
     * @see #addForce
     */
    public void clearForce(PxForceModeEnum mode) {
        checkNotNull();
        _clearForce(address, mode.value);
    }
    private static native void _clearForce(long address, int mode);

    /**
     * Clears the impulsive torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the cleared torque is to be conventional or impulsive.
     * <p>
     * Each actor has an angular acceleration and a velocity change accumulator which are directly modified using the modes PxForceMode::eACCELERATION 
     * and PxForceMode::eVELOCITY_CHANGE respectively.  The modes PxForceMode::eFORCE and PxForceMode::eIMPULSE also modify these same 
     * accumulators (see PxRigidBody::addTorque() for details); therefore the effect of calling clearTorque(PxForceMode::eFORCE) is equivalent to calling 
     * clearTorque(PxForceMode::eACCELERATION), and the effect of calling clearTorque(PxForceMode::eIMPULSE) is equivalent to calling 
     * clearTorque(PxForceMode::eVELOCITY_CHANGE). 
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * @param mode The mode to use when clearing the force/impulse(see #PxForceMode).
     * @see #addTorque
     */
    public void clearTorque(PxForceModeEnum mode) {
        checkNotNull();
        _clearTorque(address, mode.value);
    }
    private static native void _clearTorque(long address, int mode);

    /**
     * Sets the impulsive force and torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the cleared torque is to be conventional or impulsive.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * @see #addTorque
     */
    public void setForceAndTorque(PxVec3 force, PxVec3 torque) {
        checkNotNull();
        _setForceAndTorque(address, force.getAddress(), torque.getAddress());
    }
    private static native void _setForceAndTorque(long address, long force, long torque);

    /**
     * Sets the impulsive force and torque defined in the global coordinate frame to the actor.
     * <p>
     * ::PxForceMode determines if the cleared torque is to be conventional or impulsive.
     * <p>
     * <b>Note:</b> The force modes PxForceMode::eIMPULSE and PxForceMode::eVELOCITY_CHANGE can not be applied to articulation links.
     * <p>
     * <b>Note:</b> It is invalid to use this method if the actor has not been added to a scene already or if PxActorFlag::eDISABLE_SIMULATION is set.
     * @see #addTorque
     */
    public void setForceAndTorque(PxVec3 force, PxVec3 torque, PxForceModeEnum mode) {
        checkNotNull();
        _setForceAndTorque(address, force.getAddress(), torque.getAddress(), mode.value);
    }
    private static native void _setForceAndTorque(long address, long force, long torque, int mode);

    /**
     * Raises or clears a particular rigid body flag.
     * <p>
     * See the list of flags #PxRigidBodyFlag
     * <p>
     * <b>Default:</b> no flags are set
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the actor up automatically.
     * @param flag  The PxRigidBody flag to raise(set) or clear. See #PxRigidBodyFlag.
     * @param value The new boolean value for the flag.
     * @see #getRigidBodyFlags
     */
    public void setRigidBodyFlag(PxRigidBodyFlagEnum flag, boolean value) {
        checkNotNull();
        _setRigidBodyFlag(address, flag.value, value);
    }
    private static native void _setRigidBodyFlag(long address, int flag, boolean value);

    /**
     * @param inFlags WebIDL type: {@link PxRigidBodyFlags} [Ref]
     */
    public void setRigidBodyFlags(PxRigidBodyFlags inFlags) {
        checkNotNull();
        _setRigidBodyFlags(address, inFlags.getAddress());
    }
    private static native void _setRigidBodyFlags(long address, long inFlags);

    /**
     * Reads the PxRigidBody flags.
     * <p>
     * See the list of flags #PxRigidBodyFlag
     * @return The values of the PxRigidBody flags.
     * @see #setRigidBodyFlag
     */
    public PxRigidBodyFlags getRigidBodyFlags() {
        checkNotNull();
        return PxRigidBodyFlags.wrapPointer(_getRigidBodyFlags(address));
    }
    private static native long _getRigidBodyFlags(long address);

    /**
     * Sets the CCD minimum advance coefficient.
     * <p>
     * The CCD minimum advance coefficient is a value in the range [0, 1] that is used to control the minimum amount of time a body is integrated when
     * it has a CCD contact. The actual minimum amount of time that is integrated depends on various properties, including the relative speed and collision shapes
     * of the bodies involved in the contact. From these properties, a numeric value is calculated that determines the maximum distance (and therefore maximum time) 
     * which these bodies could be integrated forwards that would ensure that these bodies did not pass through each-other. This value is then scaled by CCD minimum advance
     * coefficient to determine the amount of time that will be consumed in the CCD pass.
     * <p>
     * <b>Things to consider:</b> 
     * A large value (approaching 1) ensures that the objects will always advance some time. However, larger values increase the chances of objects gently drifting through each-other in
     * scenes which the constraint solver can't converge, e.g. scenes where an object is being dragged through a wall with a constraint.
     * A value of 0 ensures that the pair of objects stop at the exact time-of-impact and will not gently drift through each-other. However, with very small/thin objects initially in 
     * contact, this can lead to a large amount of time being dropped and increases the chances of jamming. Jamming occurs when the an object is persistently in contact with an object 
     * such that the time-of-impact is 0, which results in no time being advanced for those objects in that CCD pass.
     * @param advanceCoefficient The CCD min advance coefficient. <b>Range:</b> [0, 1] <b>Default:</b> 0.15
     */
    public void setMinCCDAdvanceCoefficient(float advanceCoefficient) {
        checkNotNull();
        _setMinCCDAdvanceCoefficient(address, advanceCoefficient);
    }
    private static native void _setMinCCDAdvanceCoefficient(long address, float advanceCoefficient);

    /**
     * Gets the CCD minimum advance coefficient.
     * @return The value of the CCD min advance coefficient.
     * @see #setMinCCDAdvanceCoefficient
     */
    public float getMinCCDAdvanceCoefficient() {
        checkNotNull();
        return _getMinCCDAdvanceCoefficient(address);
    }
    private static native float _getMinCCDAdvanceCoefficient(long address);

    /**
     * Sets the maximum depenetration velocity permitted to be introduced by the solver.
     * This value controls how much velocity the solver can introduce to correct for penetrations in contacts. 
     * @param biasClamp The maximum velocity to de-penetrate by <b>Range:</b> (0, PX_MAX_F32].
     */
    public void setMaxDepenetrationVelocity(float biasClamp) {
        checkNotNull();
        _setMaxDepenetrationVelocity(address, biasClamp);
    }
    private static native void _setMaxDepenetrationVelocity(long address, float biasClamp);

    /**
     * Returns the maximum depenetration velocity the solver is permitted to introduced.
     * This value controls how much velocity the solver can introduce to correct for penetrations in contacts. 
     * @return The maximum penetration bias applied by the solver.
     */
    public float getMaxDepenetrationVelocity() {
        checkNotNull();
        return _getMaxDepenetrationVelocity(address);
    }
    private static native float _getMaxDepenetrationVelocity(long address);

    /**
     * Sets a limit on the impulse that may be applied at a contact. The maximum impulse at a contact between two dynamic or kinematic
     * bodies will be the minimum of the two limit values. For a collision between a static and a dynamic body, the impulse is limited
     * by the value for the dynamic body.
     * @param maxImpulse the maximum contact impulse. <b>Range:</b> [0, PX_MAX_F32] <b>Default:</b> PX_MAX_F32
     * @see #getMaxContactImpulse
     */
    public void setMaxContactImpulse(float maxImpulse) {
        checkNotNull();
        _setMaxContactImpulse(address, maxImpulse);
    }
    private static native void _setMaxContactImpulse(long address, float maxImpulse);

    /**
     * Returns the maximum impulse that may be applied at a contact.
     * @return The maximum impulse that may be applied at a contact
     * @see #setMaxContactImpulse
     */
    public float getMaxContactImpulse() {
        checkNotNull();
        return _getMaxContactImpulse(address);
    }
    private static native float _getMaxContactImpulse(long address);

    /**
     * Sets a distance scale whereby the angular influence of a contact on the normal constraint in a contact is 
     * zeroed if normal.cross(offset) falls below this tolerance. Rather than acting as an absolute value, this tolerance
     * is scaled by the ratio rXn.dot(angVel)/normal.dot(linVel) such that contacts that have relatively larger angular velocity
     * than linear normal velocity (e.g. rolling wheels) achieve larger slop values as the angular velocity increases.
     * @param slopCoefficient the Slop coefficient. <b>Range:</b> [0, PX_MAX_F32] <b>Default:</b> 0
     * @see #getContactSlopCoefficient
     */
    public void setContactSlopCoefficient(float slopCoefficient) {
        checkNotNull();
        _setContactSlopCoefficient(address, slopCoefficient);
    }
    private static native void _setContactSlopCoefficient(long address, float slopCoefficient);

    /**
     * Returns the contact slop coefficient.
     * @return The contact slop coefficient.
     * @see #setContactSlopCoefficient
     */
    public float getContactSlopCoefficient() {
        checkNotNull();
        return _getContactSlopCoefficient(address);
    }
    private static native float _getContactSlopCoefficient(long address);

}
