package physx.physics;

import physx.NativeObject;
import physx.common.PxRefCounted;
import physx.common.PxTransform;
import physx.geometry.PxGeometry;
import physx.support.PxMaterialPtr;

/**
 * Abstract class for collision shapes.
 * <p>
 * Shapes are shared, reference counted objects.
 * <p>
 * An instance can be created by calling the createShape() method of the PxRigidActor class, or
 * the createShape() method of the PxPhysics class.
 * <p>
 * <h3>Visualizations</h3>
 * \li PxVisualizationParameter::eCOLLISION_AABBS
 * \li PxVisualizationParameter::eCOLLISION_SHAPES
 * \li PxVisualizationParameter::eCOLLISION_AXES
 * @see PxPhysics#createShape
 * @see physx.geometry.PxBoxGeometry
 * @see physx.geometry.PxSphereGeometry
 * @see physx.geometry.PxCapsuleGeometry
 * @see physx.geometry.PxPlaneGeometry
 * @see physx.geometry.PxConvexMeshGeometry
 * PxTriangleMeshGeometry PxHeightFieldGeometry
 */
public class PxShape extends PxRefCounted {

    protected PxShape() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxShape wrapPointer(long address) {
        return address != 0L ? new PxShape(address) : null;
    }
    
    public static PxShape arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxShape(long address) {
        super(address);
    }

    // Attributes

    /**
     * *********************************************************************************************
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * *********************************************************************************************
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Adjust the geometry of the shape.
     * <p>
     * <b>Note:</b> The type of the passed in geometry must match the geometry type of the shape.
     * <b>Note:</b> It is not allowed to change the geometry type of a shape.
     * <b>Note:</b> This function does not guarantee correct/continuous behavior when objects are resting on top of old or new geometry.
     * @param geometry New geometry of the shape.
     * @see physx.geometry.PxGeometry
     */
    public void setGeometry(PxGeometry geometry) {
        checkNotNull();
        _setGeometry(address, geometry.getAddress());
    }
    private static native void _setGeometry(long address, long geometry);

    /**
     * Retrieve a reference to the shape's geometry.
     * <p>
     * \warning The returned reference has the same lifetime as the PxShape it comes from.
     * @return Reference to internal PxGeometry object.
     * @see physx.geometry.PxGeometry
     * @see #setGeometry
     */
    public PxGeometry getGeometry() {
        checkNotNull();
        return PxGeometry.wrapPointer(_getGeometry(address));
    }
    private static native long _getGeometry(long address);

    /**
     * Retrieves the actor which this shape is associated with.
     * @return The actor this shape is associated with, if it is an exclusive shape, else NULL
     * @see PxArticulationLink
     */
    public PxRigidActor getActor() {
        checkNotNull();
        return PxRigidActor.wrapPointer(_getActor(address));
    }
    private static native long _getActor(long address);

    /**
     * Assigns material(s) to the shape. Will remove existing materials from the shape.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the associated actor up automatically.
     * @param materials List of material pointers to assign to the shape. See #PxMaterial
     * @param materialCount The number of materials provided.
     * @see PxPhysics#createMaterial
     * @see #getMaterials
     */
    public void setMaterials(PxMaterialPtr materials, short materialCount) {
        checkNotNull();
        _setMaterials(address, materials.getAddress(), materialCount);
    }
    private static native void _setMaterials(long address, long materials, short materialCount);

    /**
     * Returns the number of materials assigned to the shape.
     * <p>
     * You can use #getMaterials() to retrieve the material pointers.
     * @return Number of materials associated with this shape.
     * @see PxMaterial
     * @see #getMaterials
     */
    public short getNbMaterials() {
        checkNotNull();
        return _getNbMaterials(address);
    }
    private static native short _getNbMaterials(long address);

    /**
     * Retrieve all the material pointers associated with the shape.
     * <p>
     * You can retrieve the number of material pointers by calling #getNbMaterials()
     * <p>
     * Note: The returned data may contain invalid pointers if you release materials using #PxMaterial::release().
     * @param userBuffer The buffer to store the material pointers.
     * @param bufferSize Size of provided user buffer.
     * @param startIndex Index of first material pointer to be retrieved
     * @return Number of material pointers written to the buffer.
     * @see PxMaterial
     * @see #getNbMaterials
     */
    public int getMaterials(PxMaterialPtr userBuffer, int bufferSize, int startIndex) {
        checkNotNull();
        return _getMaterials(address, userBuffer.getAddress(), bufferSize, startIndex);
    }
    private static native int _getMaterials(long address, long userBuffer, int bufferSize, int startIndex);

    /**
     * Retrieve material from given triangle index.
     * <p>
     * The input index is the internal triangle index as used inside the SDK. This is the index
     * returned to users by various SDK functions such as raycasts.
     * <p>
     * This function is only useful for triangle meshes or heightfields, which have per-triangle
     * materials. For other shapes or SDF triangle meshes, the function returns the single material
     * associated with the shape, regardless of the index.
     * @param faceIndex The internal triangle index whose material you want to retrieve.
     * @return Material from input triangle
     * <p>
     * <b>Note:</b> If faceIndex value of 0xFFFFffff is passed as an input for mesh and heightfield shapes, this function will issue a warning and return NULL.
     * <b>Note:</b> Scene queries set the value of PxQueryHit::faceIndex to 0xFFFFffff whenever it is undefined or does not apply.
     * @see PxMaterial
     * @see #getNbMaterials
     */
    public PxBaseMaterial getMaterialFromInternalFaceIndex(int faceIndex) {
        checkNotNull();
        return PxBaseMaterial.wrapPointer(_getMaterialFromInternalFaceIndex(address, faceIndex));
    }
    private static native long _getMaterialFromInternalFaceIndex(long address, int faceIndex);

    /**
     * Sets the contact offset.
     * <p>
     * Shapes whose distance is less than the sum of their contactOffset values will generate contacts. The contact offset must be positive and
     * greater than the rest offset. Having a contactOffset greater than than the restOffset allows the collision detection system to
     * predictively enforce the contact constraint even when the objects are slightly separated. This prevents jitter that would occur
     * if the constraint were enforced only when shapes were within the rest distance.
     * <p>
     * <b>Default:</b> 0.02f * PxTolerancesScale::length
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the associated actor up automatically.
     * @param contactOffset <b>Range:</b> [maximum(0,restOffset), PX_MAX_F32)
     * @see #getContactOffset
     * @see physx.common.PxTolerancesScale
     * @see #setRestOffset
     */
    public void setContactOffset(float contactOffset) {
        checkNotNull();
        _setContactOffset(address, contactOffset);
    }
    private static native void _setContactOffset(long address, float contactOffset);

    /**
     * Retrieves the contact offset. 
     * @return The contact offset of the shape.
     * @see #setContactOffset
     */
    public float getContactOffset() {
        checkNotNull();
        return _getContactOffset(address);
    }
    private static native float _getContactOffset(long address);

    /**
     * Sets the rest offset. 
     * <p>
     * Two shapes will come to rest at a distance equal to the sum of their restOffset values. If the restOffset is 0, they should converge to touching 
     * exactly.  Having a restOffset greater than zero is useful to have objects slide smoothly, so that they do not get hung up on irregularities of 
     * each others' surfaces.
     * <p>
     * <b>Default:</b> 0.0f
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the associated actor up automatically.
     * @param restOffset <b>Range:</b> (-PX_MAX_F32, contactOffset)
     * @see #getRestOffset
     * @see #setContactOffset
     */
    public void setRestOffset(float restOffset) {
        checkNotNull();
        _setRestOffset(address, restOffset);
    }
    private static native void _setRestOffset(long address, float restOffset);

    /**
     * Retrieves the rest offset. 
     * @return The rest offset of the shape.
     * @see #setRestOffset
     */
    public float getRestOffset() {
        checkNotNull();
        return _getRestOffset(address);
    }
    private static native float _getRestOffset(long address);

    /**
     * Sets torsional patch radius.
     * <p>
     * This defines the radius of the contact patch used to apply torsional friction. If the radius is 0 (and minTorsionalPatchRadius
     * is 0 too, see #setMinTorsionalPatchRadius), no torsional friction will be applied. If the radius is &gt; 0, some torsional friction
     * will be applied. This is proportional to the penetration depth so, if the shapes are separated or penetration is zero, no
     * torsional friction will be applied. It is used to approximate rotational friction introduced by the compression of contacting surfaces.
     * <p>
     * <b>Note:</b> Will only be active, if the friction patch has a single anchor point only. This is for example the case, if a contact patch
     *       has a single contact point.
     * <p>
     * <b>Note:</b> Only supported in combination with solver type PxSolverType::eTGS.
     * <p>
     * <b>Default:</b> 0.0
     * @param radius <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setTorsionalPatchRadius(float radius) {
        checkNotNull();
        _setTorsionalPatchRadius(address, radius);
    }
    private static native void _setTorsionalPatchRadius(long address, float radius);

    /**
     * Gets torsional patch radius.
     * <p>
     * See #setTorsionalPatchRadius for more info.
     * @return The torsional patch radius of the shape.
     */
    public float getTorsionalPatchRadius() {
        checkNotNull();
        return _getTorsionalPatchRadius(address);
    }
    private static native float _getTorsionalPatchRadius(long address);

    /**
     * Sets minimum torsional patch radius.
     * <p>
     * This defines the minimum radius of the contact patch used to apply torsional friction. If the radius is 0, the amount of torsional friction
     * that will be applied will be entirely dependent on the value of torsionalPatchRadius. 
     * <p>
     * If the radius is &gt; 0, some torsional friction will be applied regardless of the value of torsionalPatchRadius or the amount of penetration.
     * <p>
     * <b>Note:</b> Will only be active in certain cases, see #setTorsionalPatchRadius for details.
     * <p>
     * <b>Default:</b> 0.0
     * @param radius <b>Range:</b> [0, PX_MAX_F32)
     */
    public void setMinTorsionalPatchRadius(float radius) {
        checkNotNull();
        _setMinTorsionalPatchRadius(address, radius);
    }
    private static native void _setMinTorsionalPatchRadius(long address, float radius);

    /**
     * Gets minimum torsional patch radius.
     * <p>
     * See #setMinTorsionalPatchRadius for more info.
     * @return The minimum torsional patch radius of the shape.
     */
    public float getMinTorsionalPatchRadius() {
        checkNotNull();
        return _getMinTorsionalPatchRadius(address);
    }
    private static native float _getMinTorsionalPatchRadius(long address);

    /**
     * Sets shape flags
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the associated actor up automatically.
     * @param flag The shape flag to enable/disable. See #PxShapeFlag.
     * @param value True to set the flag. False to clear the flag specified in flag.
     * <p>
     * <b>Default:</b> PxShapeFlag::eVISUALIZATION | PxShapeFlag::eSIMULATION_SHAPE | PxShapeFlag::eSCENE_QUERY_SHAPE
     * @see #getFlags
     */
    public void setFlag(PxShapeFlagEnum flag, boolean value) {
        checkNotNull();
        _setFlag(address, flag.value, value);
    }
    private static native void _setFlag(long address, int flag, boolean value);

    /**
     * Sets shape flags
     * @see #getFlags
     */
    public void setFlags(PxShapeFlags inFlags) {
        checkNotNull();
        _setFlags(address, inFlags.getAddress());
    }
    private static native void _setFlags(long address, long inFlags);

    /**
     * Retrieves shape flags.
     * @return The values of the shape flags.
     * @see #setFlag
     */
    public PxShapeFlags getFlags() {
        checkNotNull();
        return PxShapeFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Returns true if the shape is exclusive to an actor.
     */
    public boolean isExclusive() {
        checkNotNull();
        return _isExclusive(address);
    }
    private static native boolean _isExclusive(long address);

    /**
     * Sets a name string for the object that can be retrieved with #getName().
     * <p>
     * This is for debugging and is not used by the SDK.
     * The string is not copied by the SDK, only the pointer is stored.
     * <p>
     * <b>Default:</b> NULL
     * @param name The name string to set the objects name to.
     * @see #getName
     */
    public void setName(String name) {
        checkNotNull();
        _setName(address, name);
    }
    private static native void _setName(long address, String name);

    /**
     * retrieves the name string set with setName().
     * @return The name associated with the shape.
     * @see #setName
     */
    public String getName() {
        checkNotNull();
        return _getName(address);
    }
    private static native String _getName(long address);

    /**
     * Sets the pose of the shape in actor space, i.e. relative to the actors to which they are attached.
     * <p>
     * This transformation is identity by default.
     * <p>
     * The local pose is an attribute of the shape, and so will apply to all actors to which the shape is attached.
     * <p>
     * <b>Sleeping:</b> Does <b>NOT</b> wake the associated actor up automatically.
     * <p>
     * <i>Note:</i> Does not automatically update the inertia properties of the owning actor (if applicable); use the
     * PhysX extensions method #PxRigidBodyExt::updateMassAndInertia() to do this.
     * <p>
     * <b>Default:</b> the identity transform
     * @param pose The new transform from the actor frame to the shape frame. <b>Range:</b> rigid body transform
     * @see #getLocalPose
     */
    public void setLocalPose(PxTransform pose) {
        checkNotNull();
        _setLocalPose(address, pose.getAddress());
    }
    private static native void _setLocalPose(long address, long pose);

    /**
     * Retrieves the pose of the shape in actor space, i.e. relative to the actor they are owned by.
     * <p>
     * This transformation is identity by default.
     * @return Pose of shape relative to the actor's frame.
     * @see #setLocalPose
     */
    public PxTransform getLocalPose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getLocalPose(address));
    }
    private static native long _getLocalPose(long address);

    /**
     * Sets the user definable collision filter data.
     * <p>
     * <b>Sleeping:</b> Does wake up the actor if the filter data change causes a formerly suppressed
     * collision pair to be enabled.
     * <p>
     * <b>Default:</b> (0,0,0,0)
     * @see #getSimulationFilterData
     */
    public void setSimulationFilterData(PxFilterData data) {
        checkNotNull();
        _setSimulationFilterData(address, data.getAddress());
    }
    private static native void _setSimulationFilterData(long address, long data);

    /**
     * Retrieves the shape's collision filter data.
     * @see #setSimulationFilterData
     */
    public PxFilterData getSimulationFilterData() {
        checkNotNull();
        return PxFilterData.wrapPointer(_getSimulationFilterData(address));
    }
    private static native long _getSimulationFilterData(long address);

    /**
     * Sets the user definable query filter data.
     * <p>
     * <b>Default:</b> (0,0,0,0)
     * @see #getQueryFilterData
     */
    public void setQueryFilterData(PxFilterData data) {
        checkNotNull();
        _setQueryFilterData(address, data.getAddress());
    }
    private static native void _setQueryFilterData(long address, long data);

    /**
     * Retrieves the shape's Query filter data.
     * @see #setQueryFilterData
     */
    public PxFilterData getQueryFilterData() {
        checkNotNull();
        return PxFilterData.wrapPointer(_getQueryFilterData(address));
    }
    private static native long _getQueryFilterData(long address);

}
