package physx.geomutils;

import physx.NativeObject;
import physx.common.PxBounds3;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.physics.PxHitFlags;
import physx.physics.PxRaycastHit;
import physx.physics.PxSweepHit;

public class PxGeometryQuery extends NativeObject {

    protected PxGeometryQuery() { }

    public static PxGeometryQuery wrapPointer(long address) {
        return address != 0L ? new PxGeometryQuery(address) : null;
    }
    
    protected PxGeometryQuery(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * @param unitDir  WebIDL type: {@link PxVec3} [Const, Ref]
     * @param maxDist  WebIDL type: float
     * @param geom0    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose0    WebIDL type: {@link PxTransform} [Const, Ref]
     * @param geom1    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose1    WebIDL type: {@link PxTransform} [Const, Ref]
     * @param sweepHit WebIDL type: {@link PxSweepHit} [Ref]
     * @return WebIDL type: boolean
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress());
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit);

    /**
     * @param unitDir  WebIDL type: {@link PxVec3} [Const, Ref]
     * @param maxDist  WebIDL type: float
     * @param geom0    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose0    WebIDL type: {@link PxTransform} [Const, Ref]
     * @param geom1    WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose1    WebIDL type: {@link PxTransform} [Const, Ref]
     * @param sweepHit WebIDL type: {@link PxSweepHit} [Ref]
     * @param hitFlags WebIDL type: {@link PxHitFlags} [Ref]
     * @return WebIDL type: boolean
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit, PxHitFlags hitFlags) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress(), hitFlags.getAddress());
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit, long hitFlags);

    /**
     * @param unitDir   WebIDL type: {@link PxVec3} [Const, Ref]
     * @param maxDist   WebIDL type: float
     * @param geom0     WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose0     WebIDL type: {@link PxTransform} [Const, Ref]
     * @param geom1     WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose1     WebIDL type: {@link PxTransform} [Const, Ref]
     * @param sweepHit  WebIDL type: {@link PxSweepHit} [Ref]
     * @param hitFlags  WebIDL type: {@link PxHitFlags} [Ref]
     * @param inflation WebIDL type: float
     * @return WebIDL type: boolean
     */
    public static boolean sweep(PxVec3 unitDir, float maxDist, PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1, PxSweepHit sweepHit, PxHitFlags hitFlags, float inflation) {
        return _sweep(unitDir.getAddress(), maxDist, geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress(), sweepHit.getAddress(), hitFlags.getAddress(), inflation);
    }
    private static native boolean _sweep(long unitDir, float maxDist, long geom0, long pose0, long geom1, long pose1, long sweepHit, long hitFlags, float inflation);

    /**
     * @param geom0 WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose0 WebIDL type: {@link PxTransform} [Const, Ref]
     * @param geom1 WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose1 WebIDL type: {@link PxTransform} [Const, Ref]
     * @return WebIDL type: boolean
     */
    public static boolean overlap(PxGeometry geom0, PxTransform pose0, PxGeometry geom1, PxTransform pose1) {
        return _overlap(geom0.getAddress(), pose0.getAddress(), geom1.getAddress(), pose1.getAddress());
    }
    private static native boolean _overlap(long geom0, long pose0, long geom1, long pose1);

    /**
     * @param origin   WebIDL type: {@link PxVec3} [Const, Ref]
     * @param unitDir  WebIDL type: {@link PxVec3} [Const, Ref]
     * @param geom     WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose     WebIDL type: {@link PxTransform} [Const, Ref]
     * @param maxDist  WebIDL type: float
     * @param hitFlags WebIDL type: {@link PxHitFlags} [Ref]
     * @param maxHits  WebIDL type: unsigned long
     * @param rayHits  WebIDL type: {@link PxRaycastHit}
     * @return WebIDL type: unsigned long
     */
    public static int raycast(PxVec3 origin, PxVec3 unitDir, PxGeometry geom, PxTransform pose, float maxDist, PxHitFlags hitFlags, int maxHits, PxRaycastHit rayHits) {
        return _raycast(origin.getAddress(), unitDir.getAddress(), geom.getAddress(), pose.getAddress(), maxDist, hitFlags.getAddress(), maxHits, rayHits.getAddress());
    }
    private static native int _raycast(long origin, long unitDir, long geom, long pose, float maxDist, long hitFlags, int maxHits, long rayHits);

    /**
     * @param point WebIDL type: {@link PxVec3} [Const, Ref]
     * @param geom  WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose  WebIDL type: {@link PxTransform} [Const, Ref]
     * @return WebIDL type: float
     */
    public static float pointDistance(PxVec3 point, PxGeometry geom, PxTransform pose) {
        return _pointDistance(point.getAddress(), geom.getAddress(), pose.getAddress());
    }
    private static native float _pointDistance(long point, long geom, long pose);

    /**
     * @param point        WebIDL type: {@link PxVec3} [Const, Ref]
     * @param geom         WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose         WebIDL type: {@link PxTransform} [Const, Ref]
     * @param closestPoint WebIDL type: {@link PxVec3}
     * @return WebIDL type: float
     */
    public static float pointDistance(PxVec3 point, PxGeometry geom, PxTransform pose, PxVec3 closestPoint) {
        return _pointDistance(point.getAddress(), geom.getAddress(), pose.getAddress(), closestPoint.getAddress());
    }
    private static native float _pointDistance(long point, long geom, long pose, long closestPoint);

    /**
     * @param geom WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose WebIDL type: {@link PxTransform} [Const, Ref]
     * @return WebIDL type: {@link PxBounds3} [Value]
     */
    public static PxBounds3 getWorldBounds(PxGeometry geom, PxTransform pose) {
        return PxBounds3.wrapPointer(_getWorldBounds(geom.getAddress(), pose.getAddress()));
    }
    private static native long _getWorldBounds(long geom, long pose);

    /**
     * @param geom      WebIDL type: {@link PxGeometry} [Const, Ref]
     * @param pose      WebIDL type: {@link PxTransform} [Const, Ref]
     * @param inflation WebIDL type: float
     * @return WebIDL type: {@link PxBounds3} [Value]
     */
    public static PxBounds3 getWorldBounds(PxGeometry geom, PxTransform pose, float inflation) {
        return PxBounds3.wrapPointer(_getWorldBounds(geom.getAddress(), pose.getAddress(), inflation));
    }
    private static native long _getWorldBounds(long geom, long pose, float inflation);

    /**
     * @param geom WebIDL type: {@link PxGeometry} [Const, Ref]
     * @return WebIDL type: boolean
     */
    public static boolean isValid(PxGeometry geom) {
        return _isValid(geom.getAddress());
    }
    private static native boolean _isValid(long geom);

}
