package physx.character;

import physx.NativeObject;

/**
 * Context class for obstacles.
 * <p>
 * An obstacle context class contains and manages a set of user-defined obstacles.
 * @see PxBoxObstacle
 * @see PxCapsuleObstacle
 * @see PxObstacle
 */
public class PxObstacleContext extends NativeObject {

    protected PxObstacleContext() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxObstacleContext wrapPointer(long address) {
        return address != 0L ? new PxObstacleContext(address) : null;
    }
    
    public static PxObstacleContext arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxObstacleContext(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Releases the context.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

    /**
     * Retrieves the controller manager associated with this context.
     * @return The associated controller manager
     */
    public PxControllerManager getControllerManager() {
        checkNotNull();
        return PxControllerManager.wrapPointer(_getControllerManager(address));
    }
    private static native long _getControllerManager(long address);

    /**
     * Adds an obstacle to the context.
     * @return Handle for newly-added obstacle
     */
    public int addObstacle(PxObstacle obstacle) {
        checkNotNull();
        return _addObstacle(address, obstacle.getAddress());
    }
    private static native int _addObstacle(long address, long obstacle);

    /**
     * Removes an obstacle from the context.
     * @return True if success
     */
    public boolean removeObstacle(int handle) {
        checkNotNull();
        return _removeObstacle(address, handle);
    }
    private static native boolean _removeObstacle(long address, int handle);

    /**
     * Updates data for an existing obstacle.
     * @return True if success
     */
    public boolean updateObstacle(int handle, PxObstacle obstacle) {
        checkNotNull();
        return _updateObstacle(address, handle, obstacle.getAddress());
    }
    private static native boolean _updateObstacle(long address, int handle, long obstacle);

    /**
     * Retrieves number of obstacles in the context.
     * @return Number of obstacles in the context
     */
    public int getNbObstacles() {
        checkNotNull();
        return _getNbObstacles(address);
    }
    private static native int _getNbObstacles(long address);

    /**
     * Retrieves desired obstacle.
     * @return Desired obstacle
     */
    public PxObstacle getObstacle(int i) {
        checkNotNull();
        return PxObstacle.wrapPointer(_getObstacle(address, i));
    }
    private static native long _getObstacle(long address, int i);

    /**
     * Retrieves desired obstacle by given handle.
     * @return Desired obstacle
     */
    public PxObstacle getObstacleByHandle(int handle) {
        checkNotNull();
        return PxObstacle.wrapPointer(_getObstacleByHandle(address, handle));
    }
    private static native long _getObstacleByHandle(long address, int handle);

}
