package physx.common;

import physx.NativeObject;

/**
 * Representation of a plane.
 * <p>
 *  Plane equation used: n.dot(v) + d = 0
 */
public class PxPlane extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPlane wrapPointer(long address) {
        return address != 0L ? new PxPlane(address) : null;
    }
    
    public static PxPlane arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPlane(long address) {
        super(address);
    }

    // Constructors

    /**
     * Constructor
     */
    public PxPlane() {
        address = _PxPlane();
    }
    private static native long _PxPlane();

    /**
     * Constructor from a normal and a distance
     */
    public PxPlane(float nx, float ny, float nz, float distance) {
        address = _PxPlane(nx, ny, nz, distance);
    }
    private static native long _PxPlane(float nx, float ny, float nz, float distance);

    /**
     * Constructor from a normal and a distance
     */
    public PxPlane(PxVec3 normal, float distance) {
        address = _PxPlane(normal.getAddress(), distance);
    }
    private static native long _PxPlane(long normal, float distance);

    /**
     * Constructor from three points
     */
    public PxPlane(PxVec3 p0, PxVec3 p1, PxVec3 p2) {
        address = _PxPlane(p0.getAddress(), p1.getAddress(), p2.getAddress());
    }
    private static native long _PxPlane(long p0, long p1, long p2);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The normal to the plane
     */
    public PxVec3 getN() {
        checkNotNull();
        return PxVec3.wrapPointer(_getN(address));
    }
    private static native long _getN(long address);

    /**
     * The normal to the plane
     */
    public void setN(PxVec3 value) {
        checkNotNull();
        _setN(address, value.getAddress());
    }
    private static native void _setN(long address, long value);

    /**
     * The distance from the origin
     */
    public float getD() {
        checkNotNull();
        return _getD(address);
    }
    private static native float _getD(long address);

    /**
     * The distance from the origin
     */
    public void setD(float value) {
        checkNotNull();
        _setD(address, value);
    }
    private static native void _setD(long address, float value);

    // Functions

    /**
     * @param p WebIDL type: {@link PxVec3} [Const, Ref]
     * @return WebIDL type: float
     */
    public float distance(PxVec3 p) {
        checkNotNull();
        return _distance(address, p.getAddress());
    }
    private static native float _distance(long address, long p);

    /**
     * @param p WebIDL type: {@link PxVec3} [Const, Ref]
     * @return WebIDL type: boolean
     */
    public boolean contains(PxVec3 p) {
        checkNotNull();
        return _contains(address, p.getAddress());
    }
    private static native boolean _contains(long address, long p);

    /**
     * projects p into the plane
     */
    public PxVec3 project(PxVec3 p) {
        checkNotNull();
        return PxVec3.wrapPointer(_project(address, p.getAddress()));
    }
    private static native long _project(long address, long p);

    /**
     * find an arbitrary point in the plane
     */
    public PxVec3 pointInPlane() {
        checkNotNull();
        return PxVec3.wrapPointer(_pointInPlane(address));
    }
    private static native long _pointInPlane(long address);

    /**
     * equivalent plane with unit normal
     */
    public void normalize() {
        checkNotNull();
        _normalize(address);
    }
    private static native void _normalize(long address);

    /**
     * transform plane
     */
    public PxPlane transform(PxTransform pose) {
        checkNotNull();
        return PxPlane.wrapPointer(_transform(address, pose.getAddress()));
    }
    private static native long _transform(long address, long pose);

    /**
     * inverse-transform plane
     */
    public PxPlane inverseTransform(PxTransform pose) {
        checkNotNull();
        return PxPlane.wrapPointer(_inverseTransform(address, pose.getAddress()));
    }
    private static native long _inverseTransform(long address, long pose);

}
