package physx.geometry;


/**
 * Class representing the geometry of a box.  
 * <p>
 * The geometry of a box can be fully specified by its half extents.  This is the half of its width, height, and depth.
 * <b>Note:</b> The scaling of the box is expected to be baked into these values, there is no additional scaling parameter.
 */
public class PxBoxGeometry extends PxGeometry {

    protected PxBoxGeometry() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxBoxGeometry wrapPointer(long address) {
        return address != 0L ? new PxBoxGeometry(address) : null;
    }
    
    public static PxBoxGeometry arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxBoxGeometry(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @param hx      WebIDL type: float
     * @param hy      WebIDL type: float
     * @param hz      WebIDL type: float
     * @return Stack allocated object of PxBoxGeometry
     */
    public static PxBoxGeometry createAt(long address, float hx, float hy, float hz) {
        __placement_new_PxBoxGeometry(address, hx, hy, hz);
        PxBoxGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param hx        WebIDL type: float
     * @param hy        WebIDL type: float
     * @param hz        WebIDL type: float
     * @return Stack allocated object of PxBoxGeometry
     */
    public static <T> PxBoxGeometry createAt(T allocator, Allocator<T> allocate, float hx, float hy, float hz) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxBoxGeometry(address, hx, hy, hz);
        PxBoxGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxBoxGeometry(long address, float hx, float hy, float hz);

    // Constructors

    /**
     * Constructor to initialize half extents from scalar parameters.
     * \param hx Initial half extents' x component.
     * \param hy Initial half extents' y component.
     * \param hz Initial half extents' z component.
     */
    public PxBoxGeometry(float hx, float hy, float hz) {
        address = _PxBoxGeometry(hx, hy, hz);
    }
    private static native long _PxBoxGeometry(float hx, float hy, float hz);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

}
