package physx.geometry;

import physx.NativeObject;
import physx.common.PxRefCounted;
import physx.common.PxVec3;

/**
 * A height field class.
 * <p>
 * Height fields work in a similar way as triangle meshes specified to act as
 * height fields, with some important differences:
 * <p>
 * Triangle meshes can be made of nonuniform geometry, while height fields are
 * regular, rectangular grids.  This means that with PxHeightField, you sacrifice
 * flexibility in return for improved performance and decreased memory consumption.
 * <p>
 * In local space rows extend in X direction, columns in Z direction and height in Y direction.
 * <p>
 * Like Convexes and TriangleMeshes, HeightFields are referenced by shape instances
 * (see #PxHeightFieldGeometry, #PxShape).
 * <p>
 * To avoid duplicating data when you have several instances of a particular
 * height field differently, you do not use this class to represent a
 * height field object directly. Instead, you create an instance of this height field
 * via the PxHeightFieldGeometry and PxShape classes.
 * <p>
 * <h3>Creation</h3>
 * <p>
 * To create an instance of this class call PxPhysics::createHeightField() or
 * PxCooking::createHeightField(const PxHeightFieldDesc&amp;, PxInsertionCallback&amp;).
 * To delete it call release(). This is only possible
 * once you have released all of its PxHeightFiedShape instances.
 * <p>
 * <h3>Visualizations:</h3>
 * \li #PxVisualizationParameter::eCOLLISION_AABBS
 * \li #PxVisualizationParameter::eCOLLISION_SHAPES
 * \li #PxVisualizationParameter::eCOLLISION_AXES
 * \li #PxVisualizationParameter::eCOLLISION_FNORMALS
 * \li #PxVisualizationParameter::eCOLLISION_EDGES
 * @see PxHeightFieldDesc
 * @see PxHeightFieldGeometry
 * @see physx.physics.PxShape
 * @see physx.cooking.PxCooking#createHeightField
 */
public class PxHeightField extends PxRefCounted {

    protected PxHeightField() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxHeightField wrapPointer(long address) {
        return address != 0L ? new PxHeightField(address) : null;
    }
    
    public static PxHeightField arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxHeightField(long address) {
        super(address);
    }

    // Functions

    /**
     *    Writes out the sample data array.
     * <p>
     * The user provides destBufferSize bytes storage at destBuffer.
     * The data is formatted and arranged as PxHeightFieldDesc.samples.
     * @param destBuffer The destination buffer for the sample data.
     * @param destBufferSize The size of the destination buffer.
     * @return The number of bytes written.
     */
    public int saveCells(NativeObject destBuffer, int destBufferSize) {
        checkNotNull();
        return _saveCells(address, destBuffer.getAddress(), destBufferSize);
    }
    private static native int _saveCells(long address, long destBuffer, int destBufferSize);

    /**
     *    Replaces a rectangular subfield in the sample data array.
     * <p>
     * The user provides the description of a rectangular subfield in subfieldDesc.
     * The data is formatted and arranged as PxHeightFieldDesc.samples.
     * @param startCol First cell in the destination heightfield to be modified. Can be negative.
     * @param startRow First row in the destination heightfield to be modified. Can be negative.
     * @param subfieldDesc Description of the source subfield to read the samples from.
     * @return True on success, false on failure. Failure can occur due to format mismatch.
     * <p>
     * <b>Note:</b> Modified samples are constrained to the same height quantization range as the original heightfield.
     * Source samples that are out of range of target heightfield will be clipped with no error.
     * PhysX does not keep a mapping from the heightfield to heightfield shapes that reference it.
     * Call PxShape::setGeometry on each shape which references the height field, to ensure that internal data structures are updated to reflect the new geometry.
     * Please note that PxShape::setGeometry does not guarantee correct/continuous behavior when objects are resting on top of old or new geometry.
     * @see physx.physics.PxShape#setGeometry
     */
    public boolean modifySamples(int startCol, int startRow, PxHeightFieldDesc subfieldDesc) {
        checkNotNull();
        return _modifySamples(address, startCol, startRow, subfieldDesc.getAddress());
    }
    private static native boolean _modifySamples(long address, int startCol, int startRow, long subfieldDesc);

    /**
     *    Replaces a rectangular subfield in the sample data array.
     * <p>
     * The user provides the description of a rectangular subfield in subfieldDesc.
     * The data is formatted and arranged as PxHeightFieldDesc.samples.
     * @param startCol First cell in the destination heightfield to be modified. Can be negative.
     * @param startRow First row in the destination heightfield to be modified. Can be negative.
     * @param subfieldDesc Description of the source subfield to read the samples from.
     * @param shrinkBounds If left as false, the bounds will never shrink but only grow. If set to true the bounds will be recomputed from all HF samples at O(nbColums*nbRows) perf cost.
     * @return True on success, false on failure. Failure can occur due to format mismatch.
     * <p>
     * <b>Note:</b> Modified samples are constrained to the same height quantization range as the original heightfield.
     * Source samples that are out of range of target heightfield will be clipped with no error.
     * PhysX does not keep a mapping from the heightfield to heightfield shapes that reference it.
     * Call PxShape::setGeometry on each shape which references the height field, to ensure that internal data structures are updated to reflect the new geometry.
     * Please note that PxShape::setGeometry does not guarantee correct/continuous behavior when objects are resting on top of old or new geometry.
     * @see physx.physics.PxShape#setGeometry
     */
    public boolean modifySamples(int startCol, int startRow, PxHeightFieldDesc subfieldDesc, boolean shrinkBounds) {
        checkNotNull();
        return _modifySamples(address, startCol, startRow, subfieldDesc.getAddress(), shrinkBounds);
    }
    private static native boolean _modifySamples(long address, int startCol, int startRow, long subfieldDesc, boolean shrinkBounds);

    /**
     * Retrieves the number of sample rows in the samples array.
     * @return The number of sample rows in the samples array.
     */
    public int getNbRows() {
        checkNotNull();
        return _getNbRows(address);
    }
    private static native int _getNbRows(long address);

    /**
     * Retrieves the number of sample columns in the samples array.
     * @return The number of sample columns in the samples array.
     */
    public int getNbColumns() {
        checkNotNull();
        return _getNbColumns(address);
    }
    private static native int _getNbColumns(long address);

    /**
     * Retrieves the format of the sample data.
     * @return The format of the sample data.
     */
    public PxHeightFieldFormatEnum getFormat() {
        checkNotNull();
        return PxHeightFieldFormatEnum.forValue(_getFormat(address));
    }
    private static native int _getFormat(long address);

    /**
     * Retrieves the offset in bytes between consecutive samples in the array.
     * @return The offset in bytes between consecutive samples in the array.
     */
    public int getSampleStride() {
        checkNotNull();
        return _getSampleStride(address);
    }
    private static native int _getSampleStride(long address);

    /**
     * Retrieves the convex edge threshold.
     * @return The convex edge threshold.
     */
    public float getConvexEdgeThreshold() {
        checkNotNull();
        return _getConvexEdgeThreshold(address);
    }
    private static native float _getConvexEdgeThreshold(long address);

    /**
     * Retrieves the flags bits, combined from values of the enum ::PxHeightFieldFlag.
     * @return The flags bits, combined from values of the enum ::PxHeightFieldFlag.
     */
    public PxHeightFieldFlags getFlags() {
        checkNotNull();
        return PxHeightFieldFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Retrieves the height at the given coordinates in grid space.
     * @return The height at the given coordinates or 0 if the coordinates are out of range.
     */
    public float getHeight(float x, float z) {
        checkNotNull();
        return _getHeight(address, x, z);
    }
    private static native float _getHeight(long address, float x, float z);

    /**
     * Returns material table index of given triangle
     * <p>
     * <b>Note:</b> This function takes a post cooking triangle index.
     * @param triangleIndex (internal) index of desired triangle
     * @return Material table index, or 0xffff if no per-triangle materials are used
     */
    public short getTriangleMaterialIndex(int triangleIndex) {
        checkNotNull();
        return _getTriangleMaterialIndex(address, triangleIndex);
    }
    private static native short _getTriangleMaterialIndex(long address, int triangleIndex);

    /**
     * Returns a triangle face normal for a given triangle index
     * <p>
     * <b>Note:</b> This function takes a post cooking triangle index.
     * @param triangleIndex (internal) index of desired triangle
     * @return Triangle normal for a given triangle index
     */
    public PxVec3 getTriangleNormal(int triangleIndex) {
        checkNotNull();
        return PxVec3.wrapPointer(_getTriangleNormal(address, triangleIndex));
    }
    private static native long _getTriangleNormal(long address, int triangleIndex);

    /**
     * Returns heightfield sample of given row and column 
     * @param row Given heightfield row
     * @param column Given heightfield column
     * @return Heightfield sample
     */
    public PxHeightFieldSample getSample(int row, int column) {
        checkNotNull();
        return PxHeightFieldSample.wrapPointer(_getSample(address, row, column));
    }
    private static native long _getSample(long address, int row, int column);

    /**
     * Returns the number of times the heightfield data has been modified
     * <p>
     * This method returns the number of times modifySamples has been called on this heightfield, so that code that has
     * retained state that depends on the heightfield can efficiently determine whether it has been modified.
     * @return the number of times the heightfield sample data has been modified.
     */
    public int getTimestamp() {
        checkNotNull();
        return _getTimestamp(address);
    }
    private static native int _getTimestamp(long address);

}
