package physx.physics;

import physx.NativeObject;
import physx.common.PxBase;
import physx.common.PxTransform;

/**
 * A force sensor that can be attached to articulation links to measure spatial force.
 */
public class PxArticulationSensor extends PxBase {

    protected PxArticulationSensor() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxArticulationSensor wrapPointer(long address) {
        return address != 0L ? new PxArticulationSensor(address) : null;
    }
    
    public static PxArticulationSensor arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxArticulationSensor(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public NativeObject getUserData() {
        checkNotNull();
        return NativeObject.wrapPointer(_getUserData(address));
    }
    private static native long _getUserData(long address);

    /**
     * user can assign this to whatever, usually to create a 1:1 relationship with a user object.
     */
    public void setUserData(NativeObject value) {
        checkNotNull();
        _setUserData(address, value.getAddress());
    }
    private static native void _setUserData(long address, long value);

    // Functions

    /**
     * Returns the spatial force in the local frame of the sensor.
     * @return The spatial force.
     * <p>
     * <b>Note:</b> This call is not allowed while the simulation is running except in a split simulation during #PxScene::collide() and up to #PxScene::advance(),
     * and in PxContactModifyCallback or in contact report callbacks.
     * @see #getRelativePose
     */
    public PxSpatialForce getForces() {
        checkNotNull();
        return PxSpatialForce.wrapPointer(_getForces(address));
    }
    private static native long _getForces(long address);

    /**
     * Returns the relative pose between this sensor and the body frame of the link that the sensor is attached to.
     * <p>
     * The link body frame is at the center of mass and aligned with the principal axes of inertia, see PxRigidBody::getCMassLocalPose.
     * @return The transform link body frame -&gt; sensor frame.
     * @see #setRelativePose
     */
    public PxTransform getRelativePose() {
        checkNotNull();
        return PxTransform.wrapPointer(_getRelativePose(address));
    }
    private static native long _getRelativePose(long address);

    /**
     * Sets the relative pose between this sensor and the body frame of the link that the sensor is attached to.
     * <p>
     * The link body frame is at the center of mass and aligned with the principal axes of inertia, see PxRigidBody::getCMassLocalPose.
     * @param pose The transform link body frame -&gt; sensor frame.
     * <p>
     * <b>Note:</b> Setting the sensor relative pose is not allowed while the articulation is in a scene. In order to
     * set the pose, remove and then re-add the articulation to the scene.
     * @see #getRelativePose
     */
    public void setRelativePose(PxTransform pose) {
        checkNotNull();
        _setRelativePose(address, pose.getAddress());
    }
    private static native void _setRelativePose(long address, long pose);

    /**
     * Returns the link that this sensor is attached to.
     * @return A pointer to the link.
     */
    public PxArticulationLink getLink() {
        checkNotNull();
        return PxArticulationLink.wrapPointer(_getLink(address));
    }
    private static native long _getLink(long address);

    /**
     * Returns the index of this sensor inside the articulation.
     * <p>
     * The return value is only valid for sensors attached to articulations that are in a scene.
     * @return The low-level index, or 0xFFFFFFFF if the articulation is not in a scene.
     */
    public int getIndex() {
        checkNotNull();
        return _getIndex(address);
    }
    private static native int _getIndex(long address);

    /**
     * Returns the articulation that this sensor is part of.
     * @return A pointer to the articulation.
     */
    public PxArticulationReducedCoordinate getArticulation() {
        checkNotNull();
        return PxArticulationReducedCoordinate.wrapPointer(_getArticulation(address));
    }
    private static native long _getArticulation(long address);

    /**
     * Returns the sensor's flags.
     * @return The current set of flags of the sensor.
     */
    public PxArticulationSensorFlags getFlags() {
        checkNotNull();
        return PxArticulationSensorFlags.wrapPointer(_getFlags(address));
    }
    private static native long _getFlags(long address);

    /**
     * Sets a flag of the sensor.
     * @param flag The flag to set.
     * @param enabled The value to set the flag to.
     * <p>
     * <b>Note:</b> Setting the sensor flags is not allowed while the articulation is in a scene. In order to
     * set the flags, remove and then re-add the articulation to the scene.
     */
    public void setFlag(PxArticulationSensorFlagEnum flag, boolean enabled) {
        checkNotNull();
        _setFlag(address, flag.value, enabled);
    }
    private static native void _setFlag(long address, int flag, boolean enabled);

}
