package physx.physics;

/**
 * Secondary pruning structure used for newly added objects in dynamic trees.
 * <p>
 * Dynamic trees (PxPruningStructureType::eDYNAMIC_AABB_TREE) are slowly rebuilt
 * over several frames. A secondary pruning structure holds and manages objects
 * added to the scene while this rebuild is in progress.
 * <p>
 * eNONE ignores newly added objects. This means that for a number of frames (roughly
 * defined by PxSceneQueryDesc::dynamicTreeRebuildRateHint) newly added objects will
 * be ignored by scene queries. This can be acceptable when streaming large worlds, e.g.
 * when the objects added at the boundaries of the game world don't immediately need to be
 * visible from scene queries (it would be equivalent to streaming that data in a few frames
 * later). The advantage of this approach is that there is no CPU cost associated with
 * inserting the new objects in the scene query data structures, and no extra runtime cost
 * when performing queries.
 * <p>
 * eBUCKET uses a structure similar to PxPruningStructureType::eNONE. Insertion is fast but
 * query cost can be high.
 * <p>
 * eINCREMENTAL uses an incremental AABB-tree, with no direct PxPruningStructureType equivalent.
 * Query time is fast but insertion cost can be high.
 * <p>
 * eBVH uses a PxBVH structure. This usually offers the best overall performance.
 */
public enum PxDynamicTreeSecondaryPrunerEnum {

    /**
     * no secondary pruner, new objects aren't visible to SQ for a few frames
     */
    eNONE(_geteNONE()),
    /**
     * bucket-based secondary pruner, faster updates, slower query time
     */
    eBUCKET(_geteBUCKET()),
    /**
     * incremental-BVH secondary pruner, faster query time, slower updates
     */
    eINCREMENTAL(_geteINCREMENTAL()),
    /**
     * PxBVH-based secondary pruner, good overall performance
     */
    eBVH(_geteBVH());

    static {
        de.fabmax.physxjni.Loader.load();
    }
    public final int value;
    
    PxDynamicTreeSecondaryPrunerEnum(int value) {
        this.value = value;
    }

    private static native int _geteNONE();
    private static native int _geteBUCKET();
    private static native int _geteINCREMENTAL();
    private static native int _geteBVH();
    public static PxDynamicTreeSecondaryPrunerEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxDynamicTreeSecondaryPrunerEnum: " + value);
    }

}
