package physx.physics;

/**
 * Collection of flags describing the actions to take for a collision pair.
 * <p>
 * <b>See also:</b> PxPairFlags PxSimulationFilterShader.filter() PxSimulationFilterCallback
 */
public enum PxPairFlagEnum {

    /**
     * Process the contacts of this collision pair in the dynamics solver.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     */
    eSOLVE_CONTACT(_geteSOLVE_CONTACT()),
    /**
     * Call contact modification callback for this collision pair
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>See also:</b> PxContactModifyCallback
     */
    eMODIFY_CONTACTS(_geteMODIFY_CONTACTS()),
    /**
     * Call contact report callback or trigger callback when this collision pair starts to be in contact.
     * <p>
     * If one of the two collision objects is a trigger shape (see #PxShapeFlag::eTRIGGER_SHAPE) 
     * then the trigger callback will get called as soon as the other object enters the trigger volume. 
     * If none of the two collision objects is a trigger shape then the contact report callback will get 
     * called when the actors of this collision pair start to be in contact.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact() PxSimulationEventCallback.onTrigger()
     */
    eNOTIFY_TOUCH_FOUND(_geteNOTIFY_TOUCH_FOUND()),
    /**
     * Call contact report callback while this collision pair is in contact
     * <p>
     * If none of the two collision objects is a trigger shape then the contact report callback will get 
     * called while the actors of this collision pair are in contact.
     * <p>
     * <b>Note:</b> Triggers do not support this event. Persistent trigger contacts need to be tracked separately by observing eNOTIFY_TOUCH_FOUND/eNOTIFY_TOUCH_LOST events.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> No report will get sent if the objects in contact are sleeping.
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>Note:</b> If this flag gets enabled while a pair is in touch already, there will be no eNOTIFY_TOUCH_PERSISTS events until the pair loses and regains touch.
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact() PxSimulationEventCallback.onTrigger()
     */
    eNOTIFY_TOUCH_PERSISTS(_geteNOTIFY_TOUCH_PERSISTS()),
    /**
     * Call contact report callback or trigger callback when this collision pair stops to be in contact
     * <p>
     * If one of the two collision objects is a trigger shape (see #PxShapeFlag::eTRIGGER_SHAPE) 
     * then the trigger callback will get called as soon as the other object leaves the trigger volume. 
     * If none of the two collision objects is a trigger shape then the contact report callback will get 
     * called when the actors of this collision pair stop to be in contact.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> This event will also get triggered if one of the colliding objects gets deleted.
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact() PxSimulationEventCallback.onTrigger()
     */
    eNOTIFY_TOUCH_LOST(_geteNOTIFY_TOUCH_LOST()),
    /**
     * Call contact report callback when this collision pair is in contact during CCD passes.
     * <p>
     * If CCD with multiple passes is enabled, then a fast moving object might bounce on and off the same
     * object multiple times. Hence, the same pair might be in contact multiple times during a simulation step.
     * This flag will make sure that all the detected collision during CCD will get reported. For performance
     * reasons, the system can not always tell whether the contact pair lost touch in one of the previous CCD 
     * passes and thus can also not always tell whether the contact is new or has persisted. eNOTIFY_TOUCH_CCD
     * just reports when the two collision objects were detected as being in contact during a CCD pass.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> Trigger shapes are not supported.
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact() PxSimulationEventCallback.onTrigger()
     */
    eNOTIFY_TOUCH_CCD(_geteNOTIFY_TOUCH_CCD()),
    /**
     * Call contact report callback when the contact force between the actors of this collision pair exceeds one of the actor-defined force thresholds.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact()
     */
    eNOTIFY_THRESHOLD_FORCE_FOUND(_geteNOTIFY_THRESHOLD_FORCE_FOUND()),
    /**
     * Call contact report callback when the contact force between the actors of this collision pair continues to exceed one of the actor-defined force thresholds.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> If a pair gets re-filtered and this flag has previously been disabled, then the report will not get fired in the same frame even if the force threshold has been reached in the
     * previous one (unless #eNOTIFY_THRESHOLD_FORCE_FOUND has been set in the previous frame).
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact()
     */
    eNOTIFY_THRESHOLD_FORCE_PERSISTS(_geteNOTIFY_THRESHOLD_FORCE_PERSISTS()),
    /**
     * Call contact report callback when the contact force between the actors of this collision pair falls below one of the actor-defined force thresholds (includes the case where this collision pair stops being in contact).
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies.
     * <p>
     * <b>Note:</b> If a pair gets re-filtered and this flag has previously been disabled, then the report will not get fired in the same frame even if the force threshold has been reached in the
     * previous one (unless #eNOTIFY_THRESHOLD_FORCE_FOUND or #eNOTIFY_THRESHOLD_FORCE_PERSISTS has been set in the previous frame).
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact()
     */
    eNOTIFY_THRESHOLD_FORCE_LOST(_geteNOTIFY_THRESHOLD_FORCE_LOST()),
    /**
     * Provide contact points in contact reports for this collision pair.
     * <p>
     * <b>Note:</b> Only takes effect if the colliding actors are rigid bodies and if used in combination with the flags eNOTIFY_TOUCH_... or eNOTIFY_THRESHOLD_FORCE_...
     * <p>
     * <b>Note:</b> Only takes effect if eDETECT_DISCRETE_CONTACT or eDETECT_CCD_CONTACT is raised
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact() PxContactPair PxContactPair.extractContacts()
     */
    eNOTIFY_CONTACT_POINTS(_geteNOTIFY_CONTACT_POINTS()),
    /**
     * This flag is used to indicate whether this pair generates discrete collision detection contacts. 
     * <p>
     * <b>Note:</b> Contacts are only responded to if eSOLVE_CONTACT is enabled.
     */
    eDETECT_DISCRETE_CONTACT(_geteDETECT_DISCRETE_CONTACT()),
    /**
     * This flag is used to indicate whether this pair generates CCD contacts. 
     * <p>
     * <b>Note:</b> The contacts will only be responded to if eSOLVE_CONTACT is enabled on this pair.
     * <b>Note:</b> The scene must have PxSceneFlag::eENABLE_CCD enabled to use this feature.
     * <b>Note:</b> Non-static bodies of the pair should have PxRigidBodyFlag::eENABLE_CCD specified for this feature to work correctly.
     * <b>Note:</b> This flag is not supported with trigger shapes. However, CCD trigger events can be emulated using non-trigger shapes 
     * and requesting eNOTIFY_TOUCH_FOUND and eNOTIFY_TOUCH_LOST and not raising eSOLVE_CONTACT on the pair.
     * <p>
     * <b>See also:</b> PxRigidBodyFlag::eENABLE_CCD
     * <b>See also:</b> PxSceneFlag::eENABLE_CCD
     */
    eDETECT_CCD_CONTACT(_geteDETECT_CCD_CONTACT()),
    /**
     * Provide pre solver velocities in contact reports for this collision pair.
     * <p>
     * If the collision pair has contact reports enabled, the velocities of the rigid bodies before contacts have been solved
     * will be provided in the contact report callback unless the pair lost touch in which case no data will be provided.
     * <p>
     * <b>Note:</b> Usually it is not necessary to request these velocities as they will be available by querying the velocity from the provided
     * PxRigidActor object directly. However, it might be the case that the velocity of a rigid body gets set while the simulation is running
     * in which case the PxRigidActor would return this new velocity in the contact report callback and not the velocity the simulation used.
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact(), PxContactPairVelocity, PxContactPairHeader.extraDataStream
     */
    ePRE_SOLVER_VELOCITY(_getePRE_SOLVER_VELOCITY()),
    /**
     * Provide post solver velocities in contact reports for this collision pair.
     * <p>
     * If the collision pair has contact reports enabled, the velocities of the rigid bodies after contacts have been solved
     * will be provided in the contact report callback unless the pair lost touch in which case no data will be provided.
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact(), PxContactPairVelocity, PxContactPairHeader.extraDataStream
     */
    ePOST_SOLVER_VELOCITY(_getePOST_SOLVER_VELOCITY()),
    /**
     * Provide rigid body poses in contact reports for this collision pair.
     * <p>
     * If the collision pair has contact reports enabled, the rigid body poses at the contact event will be provided 
     * in the contact report callback unless the pair lost touch in which case no data will be provided.
     * <p>
     * <b>Note:</b> Usually it is not necessary to request these poses as they will be available by querying the pose from the provided
     * PxRigidActor object directly. However, it might be the case that the pose of a rigid body gets set while the simulation is running
     * in which case the PxRigidActor would return this new pose in the contact report callback and not the pose the simulation used.
     * Another use case is related to CCD with multiple passes enabled, A fast moving object might bounce on and off the same 
     * object multiple times. This flag can be used to request the rigid body poses at the time of impact for each such collision event.
     * <p>
     * <b>See also:</b> PxSimulationEventCallback.onContact(), PxContactPairPose, PxContactPairHeader.extraDataStream
     */
    eCONTACT_EVENT_POSE(_geteCONTACT_EVENT_POSE()),
    /**
     * For internal use only.
     */
    eNEXT_FREE(_geteNEXT_FREE()),
    /**
     * Provided default flag to do simple contact processing for this collision pair.
     */
    eCONTACT_DEFAULT(_geteCONTACT_DEFAULT()),
    /**
     * Provided default flag to get commonly used trigger behavior for this collision pair.
     */
    eTRIGGER_DEFAULT(_geteTRIGGER_DEFAULT());

    static {
        de.fabmax.physxjni.Loader.load();
    }
    public final int value;
    
    PxPairFlagEnum(int value) {
        this.value = value;
    }

    private static native int _geteSOLVE_CONTACT();
    private static native int _geteMODIFY_CONTACTS();
    private static native int _geteNOTIFY_TOUCH_FOUND();
    private static native int _geteNOTIFY_TOUCH_PERSISTS();
    private static native int _geteNOTIFY_TOUCH_LOST();
    private static native int _geteNOTIFY_TOUCH_CCD();
    private static native int _geteNOTIFY_THRESHOLD_FORCE_FOUND();
    private static native int _geteNOTIFY_THRESHOLD_FORCE_PERSISTS();
    private static native int _geteNOTIFY_THRESHOLD_FORCE_LOST();
    private static native int _geteNOTIFY_CONTACT_POINTS();
    private static native int _geteDETECT_DISCRETE_CONTACT();
    private static native int _geteDETECT_CCD_CONTACT();
    private static native int _getePRE_SOLVER_VELOCITY();
    private static native int _getePOST_SOLVER_VELOCITY();
    private static native int _geteCONTACT_EVENT_POSE();
    private static native int _geteNEXT_FREE();
    private static native int _geteCONTACT_DEFAULT();
    private static native int _geteTRIGGER_DEFAULT();
    public static PxPairFlagEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxPairFlagEnum: " + value);
    }

}
