package physx.physics;

/**
 * Pruning structure used to accelerate scene queries.
 * <p>
 * eNONE uses a simple data structure that consumes less memory than the alternatives,
 * but generally has slower query performance.
 * <p>
 * eDYNAMIC_AABB_TREE usually provides the fastest queries. However there is a
 * constant per-frame management cost associated with this structure. How much work should
 * be done per frame can be tuned via the #PxSceneQueryDesc::dynamicTreeRebuildRateHint
 * parameter.
 * <p>
 * eSTATIC_AABB_TREE is typically used for static objects. It is the same as the
 * dynamic AABB tree, without the per-frame overhead. This can be a good choice for static
 * objects, if no static objects are added, moved or removed after the scene has been
 * created. If there is no such guarantee (e.g. when streaming parts of the world in and out),
 * then the dynamic version is a better choice even for static objects.
 */
public enum PxPruningStructureTypeEnum {

    /**
     * Using a simple data structure
     */
    eNONE(_geteNONE()),
    /**
     * Using a dynamic AABB tree
     */
    eDYNAMIC_AABB_TREE(_geteDYNAMIC_AABB_TREE()),
    /**
     * Using a static AABB tree
     */
    eSTATIC_AABB_TREE(_geteSTATIC_AABB_TREE());

    static {
        de.fabmax.physxjni.Loader.load();
    }
    public final int value;
    
    PxPruningStructureTypeEnum(int value) {
        this.value = value;
    }

    private static native int _geteNONE();
    private static native int _geteDYNAMIC_AABB_TREE();
    private static native int _geteSTATIC_AABB_TREE();
    public static PxPruningStructureTypeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxPruningStructureTypeEnum: " + value);
    }

}
