package physx.physics;

/**
 * Scene query update mode
 * <p>
 * This enum controls what work is done when the scene query system is updated. The updates traditionally happen when PxScene::fetchResults
 * is called. This function then calls PxSceneQuerySystem::finalizeUpdates, where the update mode is used.
 * <p>
 * fetchResults/finalizeUpdates will sync changed bounds during simulation and update the scene query bounds in pruners, this work is mandatory.
 * <p>
 * eBUILD_ENABLED_COMMIT_ENABLED does allow to execute the new AABB tree build step during fetchResults/finalizeUpdates, additionally
 * the pruner commit is called where any changes are applied. During commit PhysX refits the dynamic scene query tree and if a new tree
 * was built and the build finished the tree is swapped with current AABB tree. 
 * <p>
 * eBUILD_ENABLED_COMMIT_DISABLED does allow to execute the new AABB tree build step during fetchResults/finalizeUpdates. Pruner commit
 * is not called, this means that refit will then occur during the first scene query following fetchResults/finalizeUpdates, or may be forced
 * by the method PxScene::flushQueryUpdates() / PxSceneQuerySystemBase::flushUpdates().
 * <p>
 * eBUILD_DISABLED_COMMIT_DISABLED no further scene query work is executed. The scene queries update needs to be called manually, see
 * PxScene::sceneQueriesUpdate (see that function's doc for the equivalent PxSceneQuerySystem sequence). It is recommended to call
 * PxScene::sceneQueriesUpdate right after fetchResults/finalizeUpdates as the pruning structures are not updated.
 */
public enum PxSceneQueryUpdateModeEnum {

    /**
     * Both scene query build and commit are executed.
     */
    eBUILD_ENABLED_COMMIT_ENABLED(_geteBUILD_ENABLED_COMMIT_ENABLED()),
    /**
     * Scene query build only is executed.
     */
    eBUILD_ENABLED_COMMIT_DISABLED(_geteBUILD_ENABLED_COMMIT_DISABLED()),
    eBUILD_DISABLED_COMMIT_DISABLED(_geteBUILD_DISABLED_COMMIT_DISABLED());

    static {
        de.fabmax.physxjni.Loader.load();
    }
    public final int value;
    
    PxSceneQueryUpdateModeEnum(int value) {
        this.value = value;
    }

    private static native int _geteBUILD_ENABLED_COMMIT_ENABLED();
    private static native int _geteBUILD_ENABLED_COMMIT_DISABLED();
    private static native int _geteBUILD_DISABLED_COMMIT_DISABLED();
    public static PxSceneQueryUpdateModeEnum forValue(int value) {
        for (int i = 0; i < values().length; i++) {
            if (values()[i].value == value) {
                return values()[i];
            }
        }
        throw new IllegalArgumentException("Unknown value for enum PxSceneQueryUpdateModeEnum: " + value);
    }

}
