package physx.extensions;

import physx.NativeObject;
import physx.common.PxMat33;
import physx.common.PxQuat;
import physx.common.PxTransform;
import physx.common.PxVec3;
import physx.geometry.PxGeometry;

/**
 * Utility class to compute and manipulate mass and inertia tensor properties.
 * <p>
 * In most cases #PxRigidBodyExt::updateMassAndInertia(), #PxRigidBodyExt::setMassAndUpdateInertia() should be enough to
 * setup the mass properties of a rigid body. This utility class targets users that need to customize the mass properties
 * computation.
 */
public class PxMassProperties extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxMassProperties wrapPointer(long address) {
        return address != 0L ? new PxMassProperties(address) : null;
    }
    
    public static PxMassProperties arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxMassProperties(long address) {
        super(address);
    }

    // Constructors

    /**
     * Default constructor.
     */
    public PxMassProperties() {
        address = _PxMassProperties();
    }
    private static native long _PxMassProperties();

    /**
     * Construct from individual elements.
     */
    public PxMassProperties(float m, PxMat33 inertiaT, PxVec3 com) {
        address = _PxMassProperties(m, inertiaT.getAddress(), com.getAddress());
    }
    private static native long _PxMassProperties(float m, long inertiaT, long com);

    /**
     * Compute mass properties based on a provided geometry structure.
     * <p>
     * This constructor assumes the geometry has a density of 1. Mass and inertia tensor scale linearly with density.
     * @param geometry The geometry to compute the mass properties for. Supported geometry types are: sphere, box, capsule and convex mesh.
     */
    public PxMassProperties(PxGeometry geometry) {
        address = _PxMassProperties(geometry.getAddress());
    }
    private static native long _PxMassProperties(long geometry);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * The inertia tensor of the object.
     */
    public PxMat33 getInertiaTensor() {
        checkNotNull();
        return PxMat33.wrapPointer(_getInertiaTensor(address));
    }
    private static native long _getInertiaTensor(long address);

    /**
     * The inertia tensor of the object.
     */
    public void setInertiaTensor(PxMat33 value) {
        checkNotNull();
        _setInertiaTensor(address, value.getAddress());
    }
    private static native void _setInertiaTensor(long address, long value);

    /**
     * The center of mass of the object.
     */
    public PxVec3 getCenterOfMass() {
        checkNotNull();
        return PxVec3.wrapPointer(_getCenterOfMass(address));
    }
    private static native long _getCenterOfMass(long address);

    /**
     * The center of mass of the object.
     */
    public void setCenterOfMass(PxVec3 value) {
        checkNotNull();
        _setCenterOfMass(address, value.getAddress());
    }
    private static native void _setCenterOfMass(long address, long value);

    /**
     * The mass of the object.
     */
    public float getMass() {
        checkNotNull();
        return _getMass(address);
    }
    private static native float _getMass(long address);

    /**
     * The mass of the object.
     */
    public void setMass(float value) {
        checkNotNull();
        _setMass(address, value);
    }
    private static native void _setMass(long address, float value);

    // Functions

    /**
     * Translate the center of mass by a given vector and adjust the inertia tensor accordingly.
     * @param t The translation vector for the center of mass.
     */
    public void translate(PxVec3 t) {
        checkNotNull();
        _translate(address, t.getAddress());
    }
    private static native void _translate(long address, long t);

    /**
     * Get the entries of the diagonalized inertia tensor and the corresponding reference rotation.
     * @param inertia The inertia tensor to diagonalize.
     * @param massFrame The frame the diagonalized tensor refers to.
     * @return The entries of the diagonalized inertia tensor.
     */
    public static PxVec3 getMassSpaceInertia(PxMat33 inertia, PxQuat massFrame) {
        return PxVec3.wrapPointer(_getMassSpaceInertia(inertia.getAddress(), massFrame.getAddress()));
    }
    private static native long _getMassSpaceInertia(long inertia, long massFrame);

    /**
     * Translate an inertia tensor using the parallel axis theorem
     * @param inertia The inertia tensor to translate.
     * @param mass The mass of the object.
     * @param t The relative frame to translate the inertia tensor to.
     * @return The translated inertia tensor.
     */
    public static PxMat33 translateInertia(PxMat33 inertia, float mass, PxVec3 t) {
        return PxMat33.wrapPointer(_translateInertia(inertia.getAddress(), mass, t.getAddress()));
    }
    private static native long _translateInertia(long inertia, float mass, long t);

    /**
     * Rotate an inertia tensor around the center of mass
     * @param inertia The inertia tensor to rotate.
     * @param q The rotation from the new to the old coordinate frame, i.e. q.rotate(v) transforms
     * the coordinates of vector v from the old to the new coordinate frame.
     * @return The rotated inertia tensor.
     */
    public static PxMat33 rotateInertia(PxMat33 inertia, PxQuat q) {
        return PxMat33.wrapPointer(_rotateInertia(inertia.getAddress(), q.getAddress()));
    }
    private static native long _rotateInertia(long inertia, long q);

    /**
     * Non-uniform scaling of the inertia tensor
     * @param inertia The inertia tensor to scale.
     * @param scaleRotation The rotation from the scaling frame to the frame that inertia is expressed in.
     * I.e. scaleRotation.rotate(v) transforms the coordinates of vertex v from inertia's frame to the scaling-axes frame.
     * @param scale The scaling factor for each axis (relative to the frame specified with scaleRotation).
     * @return The scaled inertia tensor.
     */
    public static PxMat33 scaleInertia(PxMat33 inertia, PxQuat scaleRotation, PxVec3 scale) {
        return PxMat33.wrapPointer(_scaleInertia(inertia.getAddress(), scaleRotation.getAddress(), scale.getAddress()));
    }
    private static native long _scaleInertia(long inertia, long scaleRotation, long scale);

    /**
     * Sum up individual mass properties.
     * @param props Array of mass properties to sum up.
     * @param transforms Reference transforms for each mass properties entry.
     * @param count The number of mass properties to sum up.
     * @return The summed up mass properties.
     */
    public static PxMassProperties sum(PxMassProperties props, PxTransform transforms, int count) {
        return PxMassProperties.wrapPointer(_sum(props.getAddress(), transforms.getAddress(), count));
    }
    private static native long _sum(long props, long transforms, int count);

}
