package physx.geometry;


/**
 * Class representing the geometry of a capsule.
 * <p>
 * Capsules are shaped as the union of a cylinder of length 2 * halfHeight and with the 
 * given radius centered at the origin and extending along the x axis, and two hemispherical ends.
 * <b>Note:</b> The scaling of the capsule is expected to be baked into these values, there is no additional scaling parameter.
 * <p>
 * The function PxTransformFromSegment is a helper for generating an appropriate transform for the capsule from the capsule's interior line segment.
 */
public class PxCapsuleGeometry extends PxGeometry {

    protected PxCapsuleGeometry() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCapsuleGeometry wrapPointer(long address) {
        return address != 0L ? new PxCapsuleGeometry(address) : null;
    }
    
    public static PxCapsuleGeometry arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCapsuleGeometry(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address    Pre-allocated memory, where the object is created.
     * @param radius     WebIDL type: float
     * @param halfHeight WebIDL type: float
     * @return Stack allocated object of PxCapsuleGeometry
     */
    public static PxCapsuleGeometry createAt(long address, float radius, float halfHeight) {
        __placement_new_PxCapsuleGeometry(address, radius, halfHeight);
        PxCapsuleGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>        Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator  Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate   Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @param radius     WebIDL type: float
     * @param halfHeight WebIDL type: float
     * @return Stack allocated object of PxCapsuleGeometry
     */
    public static <T> PxCapsuleGeometry createAt(T allocator, Allocator<T> allocate, float radius, float halfHeight) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxCapsuleGeometry(address, radius, halfHeight);
        PxCapsuleGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxCapsuleGeometry(long address, float radius, float halfHeight);

    // Constructors

    /**
     * Constructor, initializes to a capsule with passed radius and half height.
     */
    public PxCapsuleGeometry(float radius, float halfHeight) {
        address = _PxCapsuleGeometry(radius, halfHeight);
    }
    private static native long _PxCapsuleGeometry(float radius, float halfHeight);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

}
