package physx.character;

import physx.NativeObject;
import physx.physics.PxFilterData;
import physx.physics.PxQueryFilterCallback;
import physx.physics.PxQueryFlags;

/**
 * Filtering data for "move" call.
 * <p>
 * This class contains all filtering-related parameters for the PxController::move() call.
 * <p>
 * Collisions between a CCT and the world are filtered using the mFilterData, mFilterCallback and mFilterFlags
 * members. These parameters are internally passed to PxScene::overlap() to find objects touched by the CCT.
 * Please refer to the PxScene::overlap() documentation for details.
 * <p>
 * Collisions between a CCT and another CCT are filtered using the mCCTFilterCallback member. If this filter
 * callback is not defined, none of the CCT-vs-CCT collisions are filtered, and each CCT will collide against
 * all other CCTs.
 * <p>
 * <b>Note:</b> PxQueryFlag::eANY_HIT and PxQueryFlag::eNO_BLOCK are ignored in mFilterFlags.
 * @see PxController#move
 * @see PxControllerFilterCallback
 */
public class PxControllerFilters extends NativeObject {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxControllerFilters wrapPointer(long address) {
        return address != 0L ? new PxControllerFilters(address) : null;
    }
    
    public static PxControllerFilters arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxControllerFilters(long address) {
        super(address);
    }

    // Constructors

    public PxControllerFilters() {
        address = _PxControllerFilters();
    }
    private static native long _PxControllerFilters();

    /**
     * @param filterData WebIDL type: {@link PxFilterData} [Const]
     */
    public PxControllerFilters(PxFilterData filterData) {
        address = _PxControllerFilters(filterData.getAddress());
    }
    private static native long _PxControllerFilters(long filterData);

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Attributes

    /**
     * Data for internal PxQueryFilterData structure. Passed to PxScene::overlap() call.
     */
    public PxFilterData getMFilterData() {
        checkNotNull();
        return PxFilterData.wrapPointer(_getMFilterData(address));
    }
    private static native long _getMFilterData(long address);

    /**
     * Data for internal PxQueryFilterData structure. Passed to PxScene::overlap() call.
     */
    public void setMFilterData(PxFilterData value) {
        checkNotNull();
        _setMFilterData(address, value.getAddress());
    }
    private static native void _setMFilterData(long address, long value);

    /**
     * Custom filter logic (can be NULL). Passed to PxScene::overlap() call.
     */
    public PxQueryFilterCallback getMFilterCallback() {
        checkNotNull();
        return PxQueryFilterCallback.wrapPointer(_getMFilterCallback(address));
    }
    private static native long _getMFilterCallback(long address);

    /**
     * Custom filter logic (can be NULL). Passed to PxScene::overlap() call.
     */
    public void setMFilterCallback(PxQueryFilterCallback value) {
        checkNotNull();
        _setMFilterCallback(address, value.getAddress());
    }
    private static native void _setMFilterCallback(long address, long value);

    /**
     * Flags for internal PxQueryFilterData structure. Passed to PxScene::overlap() call.
     */
    public PxQueryFlags getMFilterFlags() {
        checkNotNull();
        return PxQueryFlags.wrapPointer(_getMFilterFlags(address));
    }
    private static native long _getMFilterFlags(long address);

    /**
     * Flags for internal PxQueryFilterData structure. Passed to PxScene::overlap() call.
     */
    public void setMFilterFlags(PxQueryFlags value) {
        checkNotNull();
        _setMFilterFlags(address, value.getAddress());
    }
    private static native void _setMFilterFlags(long address, long value);

    /**
     * CCT-vs-CCT filter callback. If NULL, all CCT-vs-CCT collisions are kept.
     */
    public PxControllerFilterCallback getMCCTFilterCallback() {
        checkNotNull();
        return PxControllerFilterCallback.wrapPointer(_getMCCTFilterCallback(address));
    }
    private static native long _getMCCTFilterCallback(long address);

    /**
     * CCT-vs-CCT filter callback. If NULL, all CCT-vs-CCT collisions are kept.
     */
    public void setMCCTFilterCallback(PxControllerFilterCallback value) {
        checkNotNull();
        _setMCCTFilterCallback(address, value.getAddress());
    }
    private static native void _setMCCTFilterCallback(long address, long value);

}
