package physx.common;

import physx.NativeObject;

/**
 * Collection class for serialization.
 * <p>
 * A collection is a set of PxBase objects. PxBase objects can be added to the collection
 * regardless of other objects they depend on. Objects may be named using PxSerialObjectId values in order 
 * to resolve dependencies between objects of different collections.
 * <p>
 * Serialization and deserialization only work through collections.
 * <p>
 * A scene is typically serialized using the following steps:
 * <p>
 *  -# create a serialization registry
 *  -# create a collection for scene objects
 *  -# complete the scene objects (adds all dependent objects, e.g. meshes)
 *  -# serialize collection
 *  -# release collection
 *  -# release serialization registry
 * <p>
 * For example the code may look like this:
 * <p>
 * \code
 *  PxPhysics* physics; // The physics
 *  PxScene* scene;  // The physics scene
 *  SerialStream s;  // The user-defined stream doing the actual write to disk
 * <p>
 *  PxSerializationRegistry* registry = PxSerialization::createSerializationRegistry(*physics); // step 1)
 *  PxCollection* collection = PxSerialization::createCollection(*scene);      // step 2)
 *  PxSerialization::complete(*collection, *registry);           // step 3)
 *  PxSerialization::serializeCollectionToBinary(s, *collection, *registry);     // step 4)
 *  collection-&gt;release();                  // step 5)
 *  registry-&gt;release();                  // step 6)
 * \endcode
 * <p>
 * A scene is typically deserialized using the following steps:
 * <p>
 *  -# load a serialized collection into memory
 *  -# create a serialization registry
 *  -# create a collection by passing the serialized memory block
 *  -# add collected objects to scene
 *  -# release collection
 *  -# release serialization registry
 * <p>
 * For example the code may look like this:
 * <p>
 * \code
 *  PxPhysics* physics; // The physics
 *  PxScene* scene;  // The physics scene
 *  void* memory128; // a 128-byte aligned buffer previously loaded from disk by the user - step 1)
 * <p>
 *  PxSerializationRegistry* registry = PxSerialization::createSerializationRegistry(*physics);  // step 2)
 *  PxCollection* collection = PxSerialization::createCollectionFromBinary(memory128, *registry); // step 3)
 *  scene-&gt;addCollection(*collection);                // step 4)
 *  collection-&gt;release();                   // step 5)
 *  registry-&gt;release();                   // step 6)
 * \endcode
 */
public class PxCollection extends NativeObject {

    protected PxCollection() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxCollection wrapPointer(long address) {
        return address != 0L ? new PxCollection(address) : null;
    }
    
    public static PxCollection arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxCollection(long address) {
        super(address);
    }

    // Functions

    /**
     * Adds a PxBase object to the collection.
     * <p>
     * Adds a PxBase object to the collection. Optionally a PxSerialObjectId can be provided
     * in order to resolve dependencies between collections. A PxSerialObjectId value of PX_SERIAL_OBJECT_ID_INVALID 
     * means the object remains without id. Objects can be added regardless of other objects they require. If the object
     * is already in the collection, the ID will be set if it was PX_SERIAL_OBJECT_ID_INVALID previously, otherwise the
     * operation fails.
     */
    public void add(PxBase obj) {
        checkNotNull();
        _add(address, obj.getAddress());
    }
    private static native void _add(long address, long obj);

    /**
     * Adds a PxBase object to the collection.
     * <p>
     * Adds a PxBase object to the collection. Optionally a PxSerialObjectId can be provided
     * in order to resolve dependencies between collections. A PxSerialObjectId value of PX_SERIAL_OBJECT_ID_INVALID 
     * means the object remains without id. Objects can be added regardless of other objects they require. If the object
     * is already in the collection, the ID will be set if it was PX_SERIAL_OBJECT_ID_INVALID previously, otherwise the
     * operation fails.
     * @param id Optional PxSerialObjectId id
     */
    public void add(PxBase obj, long id) {
        checkNotNull();
        _add(address, obj.getAddress(), id);
    }
    private static native void _add(long address, long obj, long id);

    /**
     * Removes a PxBase member object from the collection.
     * <p>
     * Object needs to be contained by the collection.
     */
    public void remove(PxBase obj) {
        checkNotNull();
        _remove(address, obj.getAddress());
    }
    private static native void _remove(long address, long obj);

    /**
     * Returns whether the collection contains a certain PxBase object.
     * @return Whether object is contained.
     */
    public boolean contains(PxBase obj) {
        checkNotNull();
        return _contains(address, obj.getAddress());
    }
    private static native boolean _contains(long address, long obj);

    /**
     * Adds an id to a member PxBase object.
     * <p>
     * If the object is already associated with an id within the collection, the id is replaced.
     * May only be called for objects that are members of the collection. The id needs to be unique 
     * within the collection.
     * @param id PxSerialObjectId id to be given to the object
     */
    public void addId(PxBase obj, long id) {
        checkNotNull();
        _addId(address, obj.getAddress(), id);
    }
    private static native void _addId(long address, long obj, long id);

    /**
     * Removes id from a contained PxBase object.
     * <p>
     * May only be called for ids that are associated with an object in the collection.
     * @param id PxSerialObjectId value
     */
    public void removeId(long id) {
        checkNotNull();
        _removeId(address, id);
    }
    private static native void _removeId(long address, long id);

    /**
     * Gets number of PxBase objects in this collection.
     * @return Number of objects in this collection
     */
    public int getNbObjects() {
        checkNotNull();
        return _getNbObjects(address);
    }
    private static native int _getNbObjects(long address);

    /**
     * Gets the PxBase object of this collection given its index.
     * @param index PxBase index in [0, getNbObjects())
     * @return PxBase object at index index
     */
    public PxBase getObject(int index) {
        checkNotNull();
        return PxBase.wrapPointer(_getObject(address, index));
    }
    private static native long _getObject(long address, int index);

    /**
     * Looks for a PxBase object given a PxSerialObjectId value.
     * <p>
     * If there is no PxBase object in the collection with the given id, NULL is returned.
     * @param id PxSerialObjectId value to look for
     * @return PxBase object with the given id value or NULL
     */
    public PxBase find(long id) {
        checkNotNull();
        return PxBase.wrapPointer(_find(address, id));
    }
    private static native long _find(long address, long id);

    /**
     * Gets number of PxSerialObjectId names in this collection.
     * @return Number of PxSerialObjectId names in this collection
     */
    public int getNbIds() {
        checkNotNull();
        return _getNbIds(address);
    }
    private static native int _getNbIds(long address);

    /**
     * Gets the PxSerialObjectId name of a PxBase object within the collection.
     * <p>
     * The PxBase object needs to be a member of the collection.
     * @return PxSerialObjectId name of the object or PX_SERIAL_OBJECT_ID_INVALID if the object is unnamed
     */
    public long getId(PxBase obj) {
        checkNotNull();
        return _getId(address, obj.getAddress());
    }
    private static native long _getId(long address, long obj);

    /**
     * Deletes a collection object.
     * <p>
     * This function only deletes the collection object, i.e. the container class. It doesn't delete objects
     * that are part of the collection.
     */
    public void release() {
        checkNotNull();
        _release(address);
    }
    private static native void _release(long address);

}
