package physx.extensions;


/**
 * A fixed joint permits no relative movement between two bodies. ie the bodies are glued together.
 * <p>
 * \image html fixedJoint.png
 * @see PxJoint
 */
public class PxFixedJoint extends PxJoint {

    protected PxFixedJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxFixedJoint wrapPointer(long address) {
        return address != 0L ? new PxFixedJoint(address) : null;
    }
    
    public static PxFixedJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxFixedJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Set the linear tolerance threshold for projection. Projection is enabled if PxConstraintFlag::ePROJECTION
     * is set for the joint.
     * <p>
     * If the joint separates by more than this distance along its locked degrees of freedom, the solver 
     * will move the bodies to close the distance.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 1e10f
     * @param tolerance the linear tolerance threshold
     * @see #getProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionLinearTolerance(float tolerance) {
        checkNotNull();
        _setProjectionLinearTolerance(address, tolerance);
    }
    private static native void _setProjectionLinearTolerance(long address, float tolerance);

    /**
     * Get the linear tolerance threshold for projection.
     * @return the linear tolerance threshold
     * @see #setProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionLinearTolerance() {
        checkNotNull();
        return _getProjectionLinearTolerance(address);
    }
    private static native float _getProjectionLinearTolerance(long address);

    /**
     * Set the angular tolerance threshold for projection. Projection is enabled if 
     * PxConstraintFlag::ePROJECTION is set for the joint.
     * <p>
     * If the joint deviates by more than this angle around its locked angular degrees of freedom, 
     * the solver will move the bodies to close the angle.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0,Pi] <br>
     * <b>Default:</b> Pi
     * @param tolerance the angular tolerance threshold in radians
     * @see #getProjectionAngularTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionAngularTolerance(float tolerance) {
        checkNotNull();
        _setProjectionAngularTolerance(address, tolerance);
    }
    private static native void _setProjectionAngularTolerance(long address, float tolerance);

    /**
     * Get the angular tolerance threshold for projection.
     * @return the angular tolerance threshold in radians
     * @see #setProjectionAngularTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionAngularTolerance() {
        checkNotNull();
        return _getProjectionAngularTolerance(address);
    }
    private static native float _getProjectionAngularTolerance(long address);

}
