package physx.extensions;

import physx.NativeObject;

/**
 * Describes the parameters for a joint limit. 
 * <p>
 * Limits are enabled or disabled by setting flags or other configuration parameters joints, see the
 * documentation for specific joint types for details.
 */
public class PxJointLimitParameters extends NativeObject {

    protected PxJointLimitParameters() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxJointLimitParameters wrapPointer(long address) {
        return address != 0L ? new PxJointLimitParameters(address) : null;
    }
    
    public static PxJointLimitParameters arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxJointLimitParameters(long address) {
        super(address);
    }

    // Attributes

    /**
     * Controls the amount of bounce when the joint hits a limit.
     * <p>
     * A restitution value of 1.0 causes the joint to bounce back with the velocity which it hit the limit.
     * A value of zero causes the joint to stop dead.
     * <p>
     * In situations where the joint has many locked DOFs (e.g. 5) the restitution may not be applied 
     * correctly. This is due to a limitation in the solver which causes the restitution velocity to become zero 
     * as the solver enforces constraints on the other DOFs.
     * <p>
     * This limitation applies to both angular and linear limits, however it is generally most apparent with limited
     * angular DOFs. Disabling joint projection and increasing the solver iteration count may improve this behavior 
     * to some extent.
     * <p>
     * Also, combining soft joint limits with joint drives driving against those limits may affect stability.
     * <p>
     * <b>Range:</b> [0,1]<br>
     * <b>Default:</b> 0.0
     */
    public float getRestitution() {
        checkNotNull();
        return _getRestitution(address);
    }
    private static native float _getRestitution(long address);

    /**
     * Controls the amount of bounce when the joint hits a limit.
     * <p>
     * A restitution value of 1.0 causes the joint to bounce back with the velocity which it hit the limit.
     * A value of zero causes the joint to stop dead.
     * <p>
     * In situations where the joint has many locked DOFs (e.g. 5) the restitution may not be applied 
     * correctly. This is due to a limitation in the solver which causes the restitution velocity to become zero 
     * as the solver enforces constraints on the other DOFs.
     * <p>
     * This limitation applies to both angular and linear limits, however it is generally most apparent with limited
     * angular DOFs. Disabling joint projection and increasing the solver iteration count may improve this behavior 
     * to some extent.
     * <p>
     * Also, combining soft joint limits with joint drives driving against those limits may affect stability.
     * <p>
     * <b>Range:</b> [0,1]<br>
     * <b>Default:</b> 0.0
     */
    public void setRestitution(float value) {
        checkNotNull();
        _setRestitution(address, value);
    }
    private static native void _setRestitution(long address, float value);

    /**
     * determines the minimum impact velocity which will cause the joint to bounce
     */
    public float getBounceThreshold() {
        checkNotNull();
        return _getBounceThreshold(address);
    }
    private static native float _getBounceThreshold(long address);

    /**
     * determines the minimum impact velocity which will cause the joint to bounce
     */
    public void setBounceThreshold(float value) {
        checkNotNull();
        _setBounceThreshold(address, value);
    }
    private static native void _setBounceThreshold(long address, float value);

    /**
     * if greater than zero, the limit is soft, i.e. a spring pulls the joint back to the limit
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public float getStiffness() {
        checkNotNull();
        return _getStiffness(address);
    }
    private static native float _getStiffness(long address);

    /**
     * if greater than zero, the limit is soft, i.e. a spring pulls the joint back to the limit
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public void setStiffness(float value) {
        checkNotNull();
        _setStiffness(address, value);
    }
    private static native void _setStiffness(long address, float value);

    /**
     * if spring is greater than zero, this is the damping of the limit spring
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public float getDamping() {
        checkNotNull();
        return _getDamping(address);
    }
    private static native float _getDamping(long address);

    /**
     * if spring is greater than zero, this is the damping of the limit spring
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 0.0
     */
    public void setDamping(float value) {
        checkNotNull();
        _setDamping(address, value);
    }
    private static native void _setDamping(long address, float value);

    // Functions

    /**
     * Returns true if the current settings are valid.
     * @return true if the current settings are valid
     */
    public boolean isValid() {
        checkNotNull();
        return _isValid(address);
    }
    private static native boolean _isValid(long address);

    /**
     * @return WebIDL type: boolean
     */
    public boolean isSoft() {
        checkNotNull();
        return _isSoft(address);
    }
    private static native boolean _isSoft(long address);

}
