package physx.extensions;


/**
 * A prismatic joint permits relative translational movement between two bodies along
 * an axis, but no relative rotational movement.
 * <p>
 * the axis on each body is defined as the line containing the origin of the joint frame and
 * extending along the x-axis of that frame
 * <p>
 * \image html prismJoint.png
 * @see PxJoint
 */
public class PxPrismaticJoint extends PxJoint {

    protected PxPrismaticJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPrismaticJoint wrapPointer(long address) {
        return address != 0L ? new PxPrismaticJoint(address) : null;
    }
    
    public static PxPrismaticJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPrismaticJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * returns the displacement of the joint along its axis.
     */
    public float getPosition() {
        checkNotNull();
        return _getPosition(address);
    }
    private static native float _getPosition(long address);

    /**
     * returns the velocity of the joint along its axis
     */
    public float getVelocity() {
        checkNotNull();
        return _getVelocity(address);
    }
    private static native float _getVelocity(long address);

    /**
     * sets the joint limit  parameters.
     * <p>
     * The limit range is [-PX_MAX_F32, PX_MAX_F32], but note that the width of the limit (upper-lower) must also be
     * a valid float.
     * @see PxJointLinearLimitPair
     */
    public void setLimit(PxJointLinearLimitPair limit) {
        checkNotNull();
        _setLimit(address, limit.getAddress());
    }
    private static native void _setLimit(long address, long limit);

    /**
     * Set the flags specific to the Prismatic Joint.
     * <p>
     * <b>Default</b> PxPrismaticJointFlags(0)
     * @param flags The joint flags.
     */
    public void setPrismaticJointFlags(PxPrismaticJointFlags flags) {
        checkNotNull();
        _setPrismaticJointFlags(address, flags.getAddress());
    }
    private static native void _setPrismaticJointFlags(long address, long flags);

    /**
     * Set a single flag specific to a Prismatic Joint to true or false.
     * @param flag  The flag to set or clear.
     * @param value The value to which to set the flag
     */
    public void setPrismaticJointFlag(PxPrismaticJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setPrismaticJointFlag(address, flag.value, value);
    }
    private static native void _setPrismaticJointFlag(long address, int flag, boolean value);

    /**
     * Get the flags specific to the Prismatic Joint.
     * @return the joint flags
     */
    public PxPrismaticJointFlags getPrismaticJointFlags() {
        checkNotNull();
        return PxPrismaticJointFlags.wrapPointer(_getPrismaticJointFlags(address));
    }
    private static native long _getPrismaticJointFlags(long address);

    /**
     * Set the linear tolerance threshold for projection.
     * <p>
     * If the joint separates by more than this distance along its locked degrees of freedom, the solver 
     * will move the bodies to close the distance. 
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * This value must be nonnegative.
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 1e10f
     * @param tolerance the linear tolerance threshold
     * @see #getProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionLinearTolerance(float tolerance) {
        checkNotNull();
        _setProjectionLinearTolerance(address, tolerance);
    }
    private static native void _setProjectionLinearTolerance(long address, float tolerance);

    /**
     * Get the linear tolerance threshold for projection.
     * @return the linear tolerance threshold in radians
     * @see #setProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionLinearTolerance() {
        checkNotNull();
        return _getProjectionLinearTolerance(address);
    }
    private static native float _getProjectionLinearTolerance(long address);

    /**
     * Set the angular tolerance threshold for projection. Projection is enabled if PxConstraintFlag::ePROJECTION
     * is set for the joint.
     * <p>
     * If the joint separates by more than this distance along its locked degrees of freedom, the solver 
     * will move the bodies to close the distance.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> Pi
     * @param tolerance the linear tolerance threshold
     * @see #getProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionAngularTolerance(float tolerance) {
        checkNotNull();
        _setProjectionAngularTolerance(address, tolerance);
    }
    private static native void _setProjectionAngularTolerance(long address, float tolerance);

    /**
     * Get the angular tolerance threshold for projection.
     * @see #getProjectionAngularTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionAngularTolerance() {
        checkNotNull();
        return _getProjectionAngularTolerance(address);
    }
    private static native float _getProjectionAngularTolerance(long address);

}
