package physx.extensions;


/**
 * A joint which behaves in a similar way to a ball and socket.
 * <p>
 *  A spherical joint removes all linear degrees of freedom from two objects.
 * <p>
 *  The position of the joint on each actor is specified by the origin of the body's joint frame.
 * <p>
 *  A spherical joint may have a cone limit, to restrict the motion to within a certain range. In
 *  addition, the bodies may be projected together if the distance between them exceeds a given threshold.
 * <p>
 *  Projection, drive and limits are activated by setting the appropriate flags on the joint.
 * @see PxJoint
 */
public class PxSphericalJoint extends PxJoint {

    protected PxSphericalJoint() { }

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxSphericalJoint wrapPointer(long address) {
        return address != 0L ? new PxSphericalJoint(address) : null;
    }
    
    public static PxSphericalJoint arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxSphericalJoint(long address) {
        super(address);
    }

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

    // Functions

    /**
     * Get the limit cone.
     * @see PxJointLimitCone
     */
    public void setLimitCone(PxJointLimitCone limitCone) {
        checkNotNull();
        _setLimitCone(address, limitCone.getAddress());
    }
    private static native void _setLimitCone(long address, long limitCone);

    /**
     * get the swing angle of the joint from the Y axis
     */
    public float getSwingYAngle() {
        checkNotNull();
        return _getSwingYAngle(address);
    }
    private static native float _getSwingYAngle(long address);

    /**
     * get the swing angle of the joint from the Z axis
     */
    public float getSwingZAngle() {
        checkNotNull();
        return _getSwingZAngle(address);
    }
    private static native float _getSwingZAngle(long address);

    /**
     * Set the flags specific to the Spherical Joint.
     * <p>
     * <b>Default</b> PxSphericalJointFlags(0)
     * @param flags The joint flags.
     */
    public void setSphericalJointFlags(PxSphericalJointFlags flags) {
        checkNotNull();
        _setSphericalJointFlags(address, flags.getAddress());
    }
    private static native void _setSphericalJointFlags(long address, long flags);

    /**
     * Set a single flag specific to a Spherical Joint to true or false.
     * @param flag The flag to set or clear.
     * @param value the value to which to set the flag
     */
    public void setSphericalJointFlag(PxSphericalJointFlagEnum flag, boolean value) {
        checkNotNull();
        _setSphericalJointFlag(address, flag.value, value);
    }
    private static native void _setSphericalJointFlag(long address, int flag, boolean value);

    /**
     * Get the flags specific to the Spherical Joint.
     * @return the joint flags
     */
    public PxSphericalJointFlags getSphericalJointFlags() {
        checkNotNull();
        return PxSphericalJointFlags.wrapPointer(_getSphericalJointFlags(address));
    }
    private static native long _getSphericalJointFlags(long address);

    /**
     * Set the linear tolerance threshold for projection. Projection is enabled if PxConstraintFlag::ePROJECTION
     * is set for the joint.
     * <p>
     * If the joint separates by more than this distance along its locked degrees of freedom, the solver 
     * will move the bodies to close the distance.
     * <p>
     * Setting a very small tolerance may result in simulation jitter or other artifacts.
     * <p>
     * Sometimes it is not possible to project (for example when the joints form a cycle).
     * <p>
     * <b>Range:</b> [0, PX_MAX_F32)<br>
     * <b>Default:</b> 1e10f
     * @param tolerance the linear tolerance threshold
     * @see #getProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public void setProjectionLinearTolerance(float tolerance) {
        checkNotNull();
        _setProjectionLinearTolerance(address, tolerance);
    }
    private static native void _setProjectionLinearTolerance(long address, float tolerance);

    /**
     * Get the linear tolerance threshold for projection.
     * @return the linear tolerance threshold
     * @see #setProjectionLinearTolerance
     * @deprecated
     */
    @Deprecated
    public float getProjectionLinearTolerance() {
        checkNotNull();
        return _getProjectionLinearTolerance(address);
    }
    private static native float _getProjectionLinearTolerance(long address);

}
