package physx.geometry;


/**
 * Class describing a plane geometry.
 * <p>
 * The plane geometry specifies the half-space volume x&lt;=0. As with other geometry types, 
 * when used in a PxShape the collision volume is obtained by transforming the halfspace 
 * by the shape local pose and the actor global pose.
 * <p>
 * To generate a PxPlane from a PxTransform, transform PxPlane(1,0,0,0).
 * <p>
 * To generate a PxTransform from a PxPlane, use PxTransformFromPlaneEquation.
 * @see physx.physics.PxShape#setGeometry
 */
public class PxPlaneGeometry extends PxGeometry {

    private static native int __sizeOf();
    public static final int SIZEOF = __sizeOf();
    public static final int ALIGNOF = 8;
    
    public static PxPlaneGeometry wrapPointer(long address) {
        return address != 0L ? new PxPlaneGeometry(address) : null;
    }
    
    public static PxPlaneGeometry arrayGet(long baseAddress, int index) {
        if (baseAddress == 0L) throw new NullPointerException("baseAddress is 0");
        return wrapPointer(baseAddress + (long) SIZEOF * index);
    }
    
    protected PxPlaneGeometry(long address) {
        super(address);
    }

    // Placed Constructors

    /**
     * @param address Pre-allocated memory, where the object is created.
     * @return Stack allocated object of PxPlaneGeometry
     */
    public static PxPlaneGeometry createAt(long address) {
        __placement_new_PxPlaneGeometry(address);
        PxPlaneGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    /**
     * @param <T>       Allocator class, e.g. LWJGL's MemoryStack.
     * @param allocator Object to use for allocation, e.g. an instance of LWJGL's MemoryStack.
     * @param allocate  Method to call on allocator to obtain the target address, e.g. MemoryStack::nmalloc.
     * @return Stack allocated object of PxPlaneGeometry
     */
    public static <T> PxPlaneGeometry createAt(T allocator, Allocator<T> allocate) {
        long address = allocate.on(allocator, ALIGNOF, SIZEOF); 
        __placement_new_PxPlaneGeometry(address);
        PxPlaneGeometry createdObj = wrapPointer(address);
        createdObj.isExternallyAllocated = true;
        return createdObj;
    }

    private static native void __placement_new_PxPlaneGeometry(long address);

    // Constructors

    /**
     * Constructor.
     */
    public PxPlaneGeometry() {
        address = _PxPlaneGeometry();
    }
    private static native long _PxPlaneGeometry();

    // Destructor

    public void destroy() {
        if (address == 0L) {
            throw new IllegalStateException(this + " is already deleted");
        }
        if (isExternallyAllocated) {
            throw new IllegalStateException(this + " is externally allocated and cannot be manually destroyed");
        }
        _delete_native_instance(address);
        address = 0L;
    }
    private static native long _delete_native_instance(long address);

}
